import type { NextApiRequest, NextApiResponse } from 'next';
import { doc, updateDoc } from 'firebase/firestore';
import { db } from '@/lib/firebase'; // Assuming db is exported from firebase.ts

export default async function handler(req: NextApiRequest, res: NextApiResponse) {
  if (req.method === 'POST') {
    const { productId, trendyolUrl } = req.body;

    if (!productId || !trendyolUrl) {
      return res.status(400).json({ error: 'Product ID and Trendyol URL are required.' });
    }

    try {
      // Call the Python AI service
      const aiResponse = await fetch('http://localhost:4002/process_trendyol_url', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ url: trendyolUrl }),
      });

      if (!aiResponse.ok) {
        const errorData = await aiResponse.json();
        throw new Error(errorData.error || 'AI hizmetinden hata alındı.');
      }

      const aiResult = await aiResponse.json();

      // Update product in Firestore
      const productRef = doc(db, 'products', productId);
      await updateDoc(productRef, {
        aiScore: aiResult.ai_score || null,
        content: aiResult.description || null, // 'description' from AI maps to 'content' in Firestore
        ingredients: aiResult.ingredients || null,
        reviews: aiResult.reviews || null,
        image: aiResult.image || null, // Mapped from 'image' in AI response
        buyLink: aiResult.buyLink || null, // Mapped from 'buyLink' in AI response
        brand: aiResult.brand || null,
        category: aiResult.category || null,
        subCategory: aiResult.subCategory || null, // Mapped from 'subCategory' in AI response
        usage: aiResult.usage || null, // Mapped from 'usage' in AI response
        warnings_and_side_effects: aiResult.warnings_and_side_effects || null,
        icerik: aiResult.ingredients || null, // icerik alanı için ingredients kullan
        icerik_bilgisi: aiResult.ingredients_info || null, // icerik_bilgisi alanı için ingredients_info kullan
        form: aiResult.form || null, // New field
        katkiMaddeleri: aiResult.katkı_maddeleri || null, // New field, camelCase for Firestore
        trendyolUrl: trendyolUrl, // Save the Trendyol URL to the product document
      });

      return res.status(200).json({ message: 'Ürün başarıyla işlendi ve güncellendi.', aiResult });
    } catch (error: any) {
      console.error('Ürün işleme hatası:', error);
      return res.status(500).json({ error: (error instanceof Error) ? error.message : String(error || 'Bilinmeyen bir hata oluştu.') });
    }
  } else {
    res.setHeader('Allow', ['POST']);
    res.status(405).end(`Method ${req.method} Not Allowed`);
  }
}
