import type { Product } from "@/types"

// Form Tipi Puanları
const FORM_SCORES = {
  sivi: 75,
  kapsul: 100,
  tablet: 90,
  toz: 80,
  sase: 85,
  gummy: 70,
}

// Katkı Maddesi Ceza Puanları
const ADDITIVE_PENALTIES = {
  renklendirici: 20,
  aroma: 15,
  tatlandirici: 10,
}

// Sertifika Puanları (her biri %25)
const CERT_SCORES = {
  gmp: 25,
  haccp: 25,
  iso9001: 25,
  iso22000: 25,
}

/**
 * AI Güven Skoru - Manuel girilen değer
 */
export const getAISafetyScore = (product: Product): number => {
  return product.evaluation?.ai_guven_skoru || 0
}

/**
 * Temiz İçerik Skoru - Form tipi ve katkı maddelerine göre hesaplanır
 */
export const getCleanContentScore = (product: Product): number => {
  let score = 0

  // Form tipi puanı (sadece bir tanesi seçili olmalı)
  const formFactors = product.cleanFactors?.form
  if (formFactors) {
    const selectedForm = Object.entries(formFactors).find(([_, isSelected]) => isSelected)
    if (selectedForm) {
      const [formType] = selectedForm
      score = FORM_SCORES[formType as keyof typeof FORM_SCORES] || 0
    }
  }

  // Katkı maddesi cezaları
  const additives = product.cleanFactors?.additives
  if (additives) {
    Object.entries(additives).forEach(([additive, hasAdditive]) => {
      if (hasAdditive) {
        const penalty = ADDITIVE_PENALTIES[additive as keyof typeof ADDITIVE_PENALTIES] || 0
        score = Math.max(0, score - penalty) // Negatif olmasın
      }
    })
  }

  return score
}

/**
 * Sertifika Skoru - Her sertifika %25 puan ekler
 */
export const getCertificateScore = (product: Product): number => {
  let score = 0
  const certFactors = product.evaluation?.certFactors

  if (certFactors) {
    Object.entries(certFactors).forEach(([cert, hasCert]) => {
      if (hasCert) {
        score += CERT_SCORES[cert as keyof typeof CERT_SCORES] || 0
      }
    })
  }

  return score
}

/**
 * Toplam Güvenlik Skoru - Tüm skorların ortalaması
 */
export const getTotalSafetyScore = (product: Product): number => {
  const aiScore = getAISafetyScore(product)
  const cleanScore = getCleanContentScore(product)
  const certScore = getCertificateScore(product)

  // Ağırlıklı ortalama (AI: %40, Temiz İçerik: %40, Sertifika: %20)
  const totalScore = (aiScore * 0.4) + (cleanScore * 0.4) + (certScore * 0.2)
  
  return Math.round(totalScore)
}

/**
 * Sertifika listesini döndürür
 */
export const getCertificates = (product: Product): string[] => {
  const certFactors = product.evaluation?.certFactors
  if (!certFactors) return []

  const certificates = []
  if (certFactors.gmp) certificates.push("GMP")
  if (certFactors.haccp) certificates.push("HACCP")
  if (certFactors.iso9001) certificates.push("ISO 9001")
  if (certFactors.iso22000) certificates.push("ISO 22000")

  return certificates
}

/**
 * Skor rengini belirler
 */
export const getScoreColor = (score: number): string => {
  if (score >= 80) return "text-green-600"
  if (score >= 60) return "text-yellow-600"
  return "text-red-600"
}

/**
 * Skor badge rengini belirler
 */
export const getScoreBadgeColor = (score: number): string => {
  if (score >= 80) return "bg-green-100 text-green-800"
  if (score >= 60) return "bg-yellow-100 text-yellow-800"
  return "bg-red-100 text-red-800"
} 