// lib/firebase.ts

import { initializeApp, getApps, getApp } from 'firebase/app';
import { 
  getFirestore, 
  doc, 
  getDoc, // EKLENDİ: Tek bir dökümanı getirmek için gerekli fonksiyon.
  runTransaction, 
  collection, 
  getDocs, 
  setDoc, 
  deleteDoc, 
  query, 
  orderBy 
} from 'firebase/firestore';
import type { 
  Product, 
  Brand, 
  Category, 
  Facility, 
  HomePageSection, 
  SubCategory,
  CustomWidgetTemplate // EKLENDİ: Kullanılan type.
} from '@/types';

const firebaseConfig = {
  apiKey: process.env.NEXT_PUBLIC_FIREBASE_API_KEY,
  authDomain: process.env.NEXT_PUBLIC_FIREBASE_AUTH_DOMAIN,
  projectId: process.env.NEXT_PUBLIC_FIREBASE_PROJECT_ID,
  storageBucket: process.env.NEXT_PUBLIC_FIREBASE_STORAGE_BUCKET,
  messagingSenderId: process.env.NEXT_PUBLIC_FIREBASE_MESSAGING_SENDER_ID,
  appId: process.env.NEXT_PUBLIC_FIREBASE_APP_ID,
};

// Firebase'i başlat
const app = !getApps().length ? initializeApp(firebaseConfig) : getApp();
export const db = getFirestore(app);

// Helper to remove undefined values from an object before saving to Firestore
const removeUndefined = (obj: any): any => {
  if (typeof obj !== 'object' || obj === null) {
    return obj;
  }
  if (Array.isArray(obj)) {
    return obj.map(removeUndefined);
  }
  const newObj: { [key: string]: any } = {};
  for (const key in obj) {
    if (Object.prototype.hasOwnProperty.call(obj, key)) {
      if (obj[key] !== undefined) {
        newObj[key] = removeUndefined(obj[key]);
      }
    }
  }
  return newObj;
};

// Helper to convert Firestore Timestamp to ISO string for client-side compatibility
const convertTimestampsToISO = (data: any): any => {
  if (data === null || typeof data !== 'object') {
    return data;
  }
  if (data && typeof data.toDate === 'function') {
    return data.toDate().toISOString();
  }
  if (Array.isArray(data)) {
    return data.map(item => convertTimestampsToISO(item));
  }
  const newData: { [key: string]: any } = {};
  for (const key in data) {
    if (Object.prototype.hasOwnProperty.call(data, key)) {
      newData[key] = convertTimestampsToISO(data[key]);
    }
  }
  return newData;
};

// --- Data Loading Functions ---

export async function loadProductById(id: string): Promise<Product | null> {
  try {
    const productRef = doc(db, 'products', id);
    const productSnap = await getDoc(productRef); // Artık hata vermeyecek.

    if (productSnap.exists()) {
      const productData = productSnap.data();
      // GÜNCELLENDİ: Diğer load fonksiyonlarıyla tutarlı olması için timestamp dönüşümü eklendi.
      return convertTimestampsToISO({
        id: productSnap.id,
        ...productData,
      }) as Product;
    } else {
      console.log(`No such document with id: ${id}`);
      return null;
    }
  } catch (error) {
    console.error("Error getting product by ID:", error);
    // Hata durumunda null yerine hatayı yeniden fırlatmak, 
    // çağıran fonksiyonda daha iyi hata yönetimi sağlar.
    throw new Error(`Failed to load product with id ${id}`);
  }
}

export const loadProducts = async (): Promise<Product[]> => {
  const productsCol = collection(db, 'products');
  const snapshot = await getDocs(productsCol);
  return snapshot.docs.map(doc => convertTimestampsToISO({ id: doc.id, ...doc.data() }) as Product);
};

export const loadCategories = async (): Promise<Category[]> => {
  const categoriesCol = collection(db, 'kategoriler');
  const snapshot = await getDocs(categoriesCol);
  return snapshot.docs.map(doc => convertTimestampsToISO({ id: doc.id, ...doc.data() }) as Category);
};

export const loadBrands = async (): Promise<Brand[]> => {
  const brandsCol = collection(db, 'markalar');
  const snapshot = await getDocs(brandsCol);
  return snapshot.docs.map(doc => convertTimestampsToISO({ id: doc.id, ...doc.data() }) as Brand);
};

export const loadFacilities = async (): Promise<Facility[]> => {
  const facilitiesCol = collection(db, 'facilities');
  const snapshot = await getDocs(facilitiesCol);
  return snapshot.docs.map(doc => convertTimestampsToISO({ id: doc.id, ...doc.data() }) as Facility);
};

export const getHomePageLayout = async (): Promise<HomePageSection[]> => {
  const layoutCol = collection(db, 'homepage_layout');
  const q = query(layoutCol, orderBy('order'));
  const snapshot = await getDocs(q);
  return snapshot.docs.map(doc => convertTimestampsToISO({ id: doc.id, ...doc.data() }) as HomePageSection);
};

export const loadCustomWidgetTemplates = async (): Promise<CustomWidgetTemplate[]> => {
  const customWidgetTemplatesCol = collection(db, 'customWidgetTemplates');
  const snapshot = await getDocs(customWidgetTemplatesCol);
  return snapshot.docs.map(doc => convertTimestampsToISO({ id: doc.id, ...doc.data() }) as CustomWidgetTemplate);
};


// --- Voting and LocalStorage Functions ---

export const getLocalStorageVote = (type: 'product' | 'brand', id: string): 'like' | 'dislike' | null => {
  if (typeof window === 'undefined') return null;
  const votes = JSON.parse(localStorage.getItem('votes') || '{}');
  return votes[`${type}_${id}`] || null;
};

const setLocalStorageVote = (type: 'product' | 'brand', id: string, vote: 'like' | 'dislike' | null) => {
  if (typeof window === 'undefined') return;
  const votes = JSON.parse(localStorage.getItem('votes') || '{}');
  if (vote) {
    votes[`${type}_${id}`] = vote;
  } else {
    delete votes[`${type}_${id}`];
  }
  localStorage.setItem('votes', JSON.stringify(votes));
};

export const handleVote = async (type: 'product' | 'brand', id: string, voteType: 'like' | 'dislike') => {
  const userCurrentVote = getLocalStorageVote(type, id);
  const docRef = doc(db, `${type}s`, id);

  try {
    await runTransaction(db, async (transaction) => {
      const docSnap = await transaction.get(docRef);
      if (!docSnap.exists()) {
        throw new Error("Document does not exist!");
      }

      let newLikes = docSnap.data().likes || 0;
      let newDislikes = docSnap.data().dislikes || 0;
      let newVote: 'like' | 'dislike' | null = voteType;

      if (userCurrentVote === voteType) { // Kullanıcı aynı oyu geri alıyor
        if (voteType === 'like') newLikes--;
        else newDislikes--;
        newVote = null;
      } else if (userCurrentVote) { // Kullanıcı oyunu değiştiriyor
        if (voteType === 'like') {
          newLikes++;
          newDislikes--;
        } else {
          newLikes--;
          newDislikes++;
        }
      } else { // Kullanıcı ilk defa oy veriyor
        if (voteType === 'like') newLikes++;
        else newDislikes++;
      }

      transaction.update(docRef, { 
        likes: newLikes < 0 ? 0 : newLikes, 
        dislikes: newDislikes < 0 ? 0 : newDislikes 
      });
      setLocalStorageVote(type, id, newVote);
    });

    const finalVote = getLocalStorageVote(type, id);
    return { success: true, message: 'Oyunuz kaydedildi.', currentVote: finalVote };
  } catch (e) {
    console.error("Transaction failed: ", e);
    return { success: false, message: 'Oylama sırasında bir hata oluştu.', currentVote: userCurrentVote };
  }
};


// --- Data Writing/Admin Functions ---

export const saveProduct = async (product: Product): Promise<void> => {
  const cleanedProduct = removeUndefined(product);
  await setDoc(doc(db, 'products', cleanedProduct.id), cleanedProduct, { merge: true });
};

export const deleteProduct = async (id: string): Promise<void> => {
  await deleteDoc(doc(db, 'products', id));
};

export const saveBrand = async (brand: Brand): Promise<void> => {
  await setDoc(doc(db, 'brands', brand.id), brand, { merge: true });
};

export const deleteBrand = async (id: string): Promise<void> => {
  await deleteDoc(doc(db, 'brands', id));
};

export const saveCategory = async (category: Category): Promise<void> => {
  await setDoc(doc(db, 'kategoriler', category.id), category, { merge: true });
};

export const addSubCategory = async (categoryId: string, subCategory: SubCategory): Promise<void> => {
  const categoryRef = doc(db, 'kategoriler', categoryId);
  await runTransaction(db, async (transaction) => {
    const categoryDoc = await transaction.get(categoryRef);
    if (!categoryDoc.exists()) {
      throw new Error("Category does not exist!");
    }
    const currentSubCategories = (categoryDoc.data() as Category).subCategories || [];
    const newSubCategories = [...currentSubCategories, subCategory];
    transaction.update(categoryRef, { subCategories: newSubCategories });
  });
};

export const updateSubCategory = async (categoryId: string, subCategory: SubCategory): Promise<void> => {
  const categoryRef = doc(db, 'kategoriler', categoryId);
  await runTransaction(db, async (transaction) => {
    const categoryDoc = await transaction.get(categoryRef);
    if (!categoryDoc.exists()) {
      throw new Error("Category does not exist!");
    }
    const currentSubCategories = (categoryDoc.data() as Category).subCategories || [];
    const newSubCategories = currentSubCategories.map(sub => 
      sub.id === subCategory.id ? subCategory : sub
    );
    transaction.update(categoryRef, { subCategories: newSubCategories });
  });
};

export const deleteSubCategory = async (categoryId: string, subCategoryId: string): Promise<void> => {
  const categoryRef = doc(db, 'kategoriler', categoryId);
  await runTransaction(db, async (transaction) => {
    const categoryDoc = await transaction.get(categoryRef);
    if (!categoryDoc.exists()) {
      throw new Error("Category does not exist!");
    }
    const currentSubCategories = (categoryDoc.data() as Category).subCategories || [];
    const newSubCategories = currentSubCategories.filter(sub => sub.id !== subCategoryId);
    transaction.update(categoryRef, { subCategories: newSubCategories });
  });
};

export const deleteCategory = async (id: string): Promise<void> => {
  await deleteDoc(doc(db, 'categories', id));
};

export const saveFacility = async (facility: Facility): Promise<void> => {
  const cleanedFacility = removeUndefined(facility);
  await setDoc(doc(db, 'facilities', cleanedFacility.id), cleanedFacility, { merge: true });
};

export const deleteFacility = async (id: string): Promise<void> => {
  await deleteDoc(doc(db, 'facilities', id));
};

export const saveHomePageLayout = async (layout: HomePageSection[]): Promise<void> => {
  const batch = runTransaction(db, async (transaction) => {
    const layoutColRef = collection(db, 'homepage_layout');
    const snapshot = await getDocs(layoutColRef);
    snapshot.docs.forEach(d => transaction.delete(d.ref));
    
    layout.forEach((section, index) => {
      const newDocRef = doc(collection(db, 'homepage_layout'));
      transaction.set(newDocRef, { ...section, id: newDocRef.id, order: index });
    });
  });
  await batch;
};

export const saveCustomWidgetTemplate = async (template: CustomWidgetTemplate): Promise<void> => {
  await setDoc(doc(db, 'customWidgetTemplates', template.id), template, { merge: true });
};

export const deleteCustomWidgetTemplate = async (id: string): Promise<void> => {
  await deleteDoc(doc(db, 'customWidgetTemplates', id));
};