"use client"

import { useState, useEffect, useCallback } from "react"
import { Search, Filter, Edit, Trash2, Plus } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import type { Product, Category } from "@/types"
import Image from "next/image"
import { deleteProduct, saveProduct } from "@/lib/firebase"
import { Checkbox } from "@/components/ui/checkbox"

interface ProductsTabProps {
  products: Product[]
  loading: boolean
  search: string
  category: string
  sort: string
  setSearch: (search: string) => void
  setCategory: (category: string) => void
  setSort: (sort: string) => void
  onEdit: (product: Product) => void
  onDelete: (id: string) => void
  onAdd: () => void
  onRefresh: () => void
  categories?: Category[]
}

const QUICK_EDIT_FIELDS = [
  { value: "name", label: "Ürün Adı" },
  { value: "image", label: "Fotoğraf URL" },
  { value: "onayNo", label: "Onay Numarası" },
  // Diğer alanlar eklenebilir
];

// Type-safe field getter for quick edit
function getQuickEditValue(product: Product, field: string): string {
  if (field === "name") return product.name;
  if (field === "image") return product.image;
  if (field === "onayNo") return product.onayNo || "";
  return "";
}
function setQuickEditValue(product: Product, field: string, value: string): Partial<Product> {
  if (field === "name") return { name: value };
  if (field === "image") return { image: value };
  if (field === "onayNo") return { onayNo: value };
  return {};
}

export default function ProductsTab({
  products,
  loading,
  search,
  category,
  sort,
  setSearch,
  setCategory,
  setSort,
  onEdit,
  onDelete,
  onAdd,
  onRefresh,
  categories = [],
}: ProductsTabProps) {
  const [filteredProducts, setFilteredProducts] = useState<Product[]>([])
  const [deleting, setDeleting] = useState<string | null>(null)
  const [selectedIds, setSelectedIds] = useState<string[]>([])
  const [quickEditField, setQuickEditField] = useState<string>("name");
  const [quickEditValues, setQuickEditValues] = useState<Record<string, string>>({});
  const [savingQuickEdit, setSavingQuickEdit] = useState(false);

  // Kategori adını bulma fonksiyonu
  const getCategoryName = useCallback((categoryId: string) => {
    return categories.find(cat => cat.id === categoryId)?.name || categoryId
  }, [categories]);

  useEffect(() => {
    let filtered = [...products]

    // Search filter
    if (search) {
      const searchLower = search.toLowerCase()
      filtered = filtered.filter(
        (p) =>
          p.name.toLowerCase().includes(searchLower) ||
          (typeof p.brand === 'string' ? p.brand.toLowerCase() : p.brand.name.toLowerCase()).includes(searchLower) ||
          getCategoryName(p.category).toLowerCase().includes(searchLower),
      )
    }

    // Category filter
    if (category && category !== "all") {
      filtered = filtered.filter((p) => p.category === category)
    }

    // Sort
    filtered.sort((a, b) => {
      switch (sort) {
        case "name-asc":
          return a.name.localeCompare(b.name)
        case "name-desc":
          return b.name.localeCompare(a.name)
        case "likes-desc":
          return b.likes - a.likes
        case "likes-asc":
          return a.likes - b.likes
        case "date-desc":
          return new Date(b.createdAt || 0).getTime() - new Date(a.createdAt || 0).getTime()
        case "date-asc":
          return new Date(a.createdAt || 0).getTime() - new Date(b.createdAt || 0).getTime()
        default:
          return 0
      }
    })

    setFilteredProducts(filtered)
  }, [products, search, category, sort, categories, getCategoryName])

  const handleDelete = async (id: string) => {
    if (!confirm("Bu ürünü silmek istediğinizden emin misiniz?")) return

    try {
      setDeleting(id)
      await deleteProduct(id)
      onDelete(id)
      onRefresh() // Listeyi yenile
    } catch (error) {
      console.error("Ürün silinirken hata:", error)
      alert("Ürün silinirken bir hata oluştu.")
    } finally {
      setDeleting(null)
    }
  }

  const handleSelect = (id: string) => {
    setSelectedIds((prev) => prev.includes(id) ? prev.filter((sid) => sid !== id) : [...prev, id])
  }

  const handleSelectAll = () => {
    if (selectedIds.length === filteredProducts.length) {
      setSelectedIds([])
    } else {
      setSelectedIds(filteredProducts.map((p) => p.id))
    }
  }

  const handleBulkDelete = async () => {
    if (!selectedIds.length) return
    if (!confirm(`${selectedIds.length} ürünü silmek istediğinize emin misiniz?`)) return
    try {
      await Promise.all(selectedIds.map(async (id) => {
        await deleteProduct(id)
        onDelete(id)
      }))
      setSelectedIds([])
      onRefresh()
    } catch (error) {
      alert("Toplu silme sırasında hata oluştu.")
    }
  }

  // Hızlı düzenle kaydet
  const handleQuickEditSave = async () => {
    setSavingQuickEdit(true);
    try {
      await Promise.all(
        Object.entries(quickEditValues).map(async ([id, value]) => {
          const product = products.find(p => p.id === id);
          if (!product) return;
          // Sadece değişiklik varsa kaydet
          if (getQuickEditValue(product, quickEditField) !== value) {
            // Firestore'a güncelleme fonksiyonunu burada çağırmalısın
            await saveProduct({ ...product, ...setQuickEditValue(product, quickEditField, value) });
          }
        })
      );
      setQuickEditValues({});
      onRefresh();
    } catch (e) {
      alert("Hızlı düzenleme kaydedilirken hata oluştu.");
    } finally {
      setSavingQuickEdit(false);
    }
  };

  if (loading) {
    return (
      <Card>
        <CardContent className="flex items-center justify-center py-20">
          <div className="text-center">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto mb-4"></div>
            <p className="text-gray-600">Ürünler yükleniyor...</p>
          </div>
        </CardContent>
      </Card>
    )
  }

  return (
    <div className="space-y-6">
      
      {/* Filters */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Filter className="h-5 w-5" />
            Filtreler
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex flex-col md:flex-row gap-4">
            <div className="flex-1">
              <div className="relative">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                <Input
                  placeholder="Ürün adı, marka veya kategori ara..."
                  value={search}
                  onChange={(e) => setSearch(e.target.value)}
                  className="pl-10"
                />
              </div>
            </div>

            <Select value={category === "" ? "all" : category} onValueChange={setCategory}>
              <SelectTrigger className="w-full md:w-48">
                <SelectValue placeholder="Tüm Kategoriler" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">Tüm Kategoriler</SelectItem>
                {categories.map((category) => (
                  <SelectItem key={category.id} value={category.id}>
                    {category.name}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>

            <Select value={sort === "" ? "name-asc" : sort} onValueChange={setSort}>
              <SelectTrigger className="w-full md:w-48">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="name-asc">İsim (A-Z)</SelectItem>
                <SelectItem value="name-desc">İsim (Z-A)</SelectItem>
                <SelectItem value="likes-desc">Beğeni (Yüksek-Düşük)</SelectItem>
                <SelectItem value="likes-asc">Beğeni (Düşük-Yüksek)</SelectItem>
                <SelectItem value="date-desc">Yeni Eklenenler</SelectItem>
                <SelectItem value="date-asc">Eski Eklenenler</SelectItem>
              </SelectContent>
            </Select>
          </div>
        </CardContent>
      </Card>

      {/* Products List */}
      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <CardTitle>Ürünler ({filteredProducts.length})</CardTitle>
            <div className="flex gap-2">
              <Button variant="destructive" onClick={handleBulkDelete} disabled={!selectedIds.length}>
                Seçilenleri Sil
              </Button>
            <Select value={quickEditField} onValueChange={setQuickEditField}>
              <SelectTrigger className="w-[180px]">
                <SelectValue placeholder="Hızlı Düzenle" />
              </SelectTrigger>
              <SelectContent>
                {QUICK_EDIT_FIELDS.map((field) => (
                  <SelectItem key={field.value} value={field.value}>
                    {field.label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
            {quickEditField && (
              <Button onClick={handleQuickEditSave} disabled={savingQuickEdit}>
                {savingQuickEdit ? "Kaydediliyor..." : "Hızlı Kaydet"}
              </Button>
            )}
            <Button onClick={onAdd}>
              <Plus className="h-4 w-4 mr-2" />
              Yeni Ürün
            </Button>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          {filteredProducts.length === 0 ? (
            <div className="text-center py-12">
              <p className="text-gray-500">Filtreye uygun ürün bulunamadı.</p>
            </div>
          ) : (
            <div className="space-y-4">
              {/* Desktop Table Header */}
              <div className="hidden md:grid grid-cols-12 gap-3 p-3 bg-gray-50 rounded-lg font-medium text-sm text-gray-700">
                <div className="col-span-1 flex items-center">
                  <Checkbox
                    checked={selectedIds.length === filteredProducts.length && filteredProducts.length > 0
                      ? true
                      : selectedIds.length > 0
                      ? 'indeterminate'
                      : false}
                    onCheckedChange={handleSelectAll}
                  />
                </div>
                <div className="col-span-3">Ürün Adı</div>
                <div className="col-span-3">Marka</div>
                <div className="col-span-2">Kategori</div>
                <div className="col-span-1">Alt Kategori</div>
                <div className="col-span-1">👍</div>
                <div className="col-span-2">İşlemler</div>
                {quickEditField && <div className="col-span-2">Hızlı Düzenle ({QUICK_EDIT_FIELDS.find(f => f.value === quickEditField)?.label})</div>}
              </div>

              {/* Products */}
              <div className="space-y-2">
                {filteredProducts.map((product, index) => (
                  <div
                    key={product.id}
                    className="grid grid-cols-1 md:grid-cols-12 gap-3 p-3 bg-white border rounded-lg hover:shadow-md transition-all duration-200 animate-in fade-in slide-in-from-bottom-2"
                    style={{ animationDelay: `${index * 50}ms` }}
                  >
                    <div className="md:col-span-1 flex items-center">
                      <Checkbox checked={selectedIds.includes(product.id)} onCheckedChange={() => handleSelect(product.id)} />
                    </div>
                    {/* Mobile Layout */}
                    <div className="md:hidden space-y-3">
                      <div className="flex items-start gap-3">
                        <div className="w-14 h-14 bg-gray-100 rounded-lg flex items-center justify-center overflow-hidden">
                          <Image
                            src={product.image || "/placeholder.svg"}
                            alt={product.name}
                            width={56}
                            height={56}
                            className="object-contain"
                          />
                        </div>
                        <div className="flex-1 min-w-0">
                          <h3 className="font-medium text-gray-900 truncate text-sm">{product.name}</h3>
                          <div className="text-xs text-gray-600">
                            {typeof product.brand === 'string' ? product.brand : product.brand?.name}
                          </div>
                          <div className="flex items-center gap-2 mt-1">
                            <Badge variant="secondary" className="text-xs">
                              {getCategoryName(product.category)}
                            </Badge>
                            {product.subCategory && (
                              <Badge variant="outline" className="text-xs">
                                {product.subCategory}
                              </Badge>
                            )}
                          </div>
                        </div>
                        <div className="flex items-center gap-1 text-xs text-gray-600">👍 {product.likes}</div>
                      </div>

                      {/* AI Score */}
                      {product.evaluation?.ai_guven_skoru && (
                        <div className="bg-gradient-to-r from-purple-50 to-blue-50 p-2 rounded-lg">
                          <div className="flex items-center justify-between">
                            <span className="text-xs font-medium text-gray-700">🤖 AI Güven Skoru</span>
                            <span className="text-xs font-bold text-blue-600">{product.evaluation.ai_guven_skoru}%</span>
                          </div>
                          <div className="w-full bg-gray-200 rounded-full h-1.5 mt-1">
                            <div
                              className="bg-gradient-to-r from-blue-500 to-purple-500 h-1.5 rounded-full transition-all duration-500"
                              style={{ width: `${product.evaluation.ai_guven_skoru}%` }}
                            />
                          </div>
                        </div>
                      )}

                      <div className="flex items-center gap-2">
                        <Button size="sm" variant="outline" onClick={() => onEdit(product)} className="h-8 px-2">
                          <Edit className="h-3 w-3 mr-1" />
                          <span className="text-xs">Düzenle</span>
                        </Button>
                        <Button
                          size="sm"
                          variant="outline"
                          className="text-red-600 hover:text-red-700 h-8 px-2"
                          onClick={() => handleDelete(product.id)}
                          disabled={deleting === product.id}
                        >
                          <Trash2 className="h-3 w-3 mr-1" />
                          <span className="text-xs">{deleting === product.id ? "Siliniyor..." : "Sil"}</span>
                        </Button>
                      </div>
                    </div>

                    {/* Desktop Layout */}
                    <div className="hidden md:contents">
                      <div className="col-span-1 flex items-center">
                        <div className="w-10 h-10 bg-gray-100 rounded-lg flex items-center justify-center overflow-hidden">
                          <Image
                            src={product.image || "/placeholder.svg"}
                            alt={product.name}
                            width={40}
                            height={40}
                            className="object-contain"
                          />
                        </div>
                      </div>

                      <div className="col-span-3 flex items-center">
                        <div>
                          <h3 className="font-medium text-gray-900 truncate text-sm">{product.name}</h3>
                          {product.evaluation?.ai_guven_skoru && (
                            <div className="flex items-center gap-2 mt-1">
                              <span className="text-xs text-gray-500">AI Güven:</span>
                              <span className="text-xs font-bold text-blue-600">{product.evaluation.ai_guven_skoru}%</span>
                              <div className="w-10 bg-gray-200 rounded-full h-1">
                                <div
                                  className="bg-blue-500 h-1 rounded-full"
                                  style={{ width: `${product.evaluation.ai_guven_skoru}%` }}
                                />
                              </div>
                            </div>
                          )}
                        </div>
                      </div>

                      <div className="col-span-3 flex items-center">
                        <span className="text-gray-900 text-sm">{typeof product.brand === 'string' ? product.brand : product.brand?.name}</span>
                      </div>

                      <div className="col-span-2 flex items-center">
                        <Badge variant="secondary" className="text-xs">{getCategoryName(product.category)}</Badge>
                      </div>

                      <div className="col-span-1 flex items-center">
                        {product.subCategory && (
                          <Badge variant="outline" className="text-xs">
                            {product.subCategory}
                          </Badge>
                        )}
                      </div>

                      <div className="col-span-1 flex items-center">
                        <span className="text-gray-900 text-sm">{product.likes}</span>
                      </div>

                      <div className="col-span-2 flex items-center gap-2">
                        <Button size="sm" variant="outline" onClick={() => onEdit(product)} className="h-8 px-2">
                          <Edit className="h-3 w-3" />
                        </Button>
                        <Button
                          size="sm"
                          variant="outline"
                          className="text-red-600 hover:text-red-700 h-8 px-2"
                          onClick={() => handleDelete(product.id)}
                          disabled={deleting === product.id}
                        >
                          <Trash2 className="h-3 w-3" />
                          {deleting === product.id ? "Siliniyor..." : ""}
                        </Button>
                      </div>
                      {quickEditField && (
                        <div className="col-span-2 flex items-center">
                          <Input
                            type="text"
                            className="text-xs"
                            value={quickEditValues[product.id] ?? getQuickEditValue(product, quickEditField) ?? ""}
                            onChange={e => setQuickEditValues(v => ({ ...v, [product.id]: e.target.value }))}
                            placeholder={QUICK_EDIT_FIELDS.find(f => f.value === quickEditField)?.label}
                          />
                        </div>
                      )}
                    </div>
                    {/* Mobile Layout için de eklenebilir */}
                    {quickEditField && (
                      <div className="md:hidden mt-2">
                        <Input
                          type="text"
                          className="text-xs"
                          value={quickEditValues[product.id] ?? getQuickEditValue(product, quickEditField) ?? ""}
                          onChange={e => setQuickEditValues(v => ({ ...v, [product.id]: e.target.value }))}
                          placeholder={QUICK_EDIT_FIELDS.find(f => f.value === quickEditField)?.label}
                        />
                      </div>
                    )}
                  </div>
                ))}
              </div>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  )
}
