"use client"

import { useState } from "react";
import { Product } from "@/types";
import { Button } from "@/components/ui/button";
import { Command, CommandEmpty, CommandGroup, CommandInput, CommandItem, CommandList } from "@/components/ui/command";
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover";
import { Check, ChevronsUpDown, Plus, Trash2, X } from "lucide-react";
import { cn } from "@/lib/utils";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";

interface ProductSelectProps {
  allProducts: Product[];
  selectedProductIds: string[];
  onProductIdsChange: (ids: string[]) => void;
}

const ProductSelect = ({
  allProducts,
  selectedProductIds,
  onProductIdsChange,
}: ProductSelectProps) => {
  const [open, setOpen] = useState(false);
  const [searchValue, setSearchValue] = useState("");

  const handleSelect = (productId: string) => {
    const newSelectedIds = selectedProductIds.includes(productId)
      ? selectedProductIds.filter((id) => id !== productId)
      : [...selectedProductIds, productId];
    onProductIdsChange(newSelectedIds);
  };

  const getProductName = (productId: string) => {
    return allProducts.find(p => p.id === productId)?.name || "Bilinmeyen Ürün";
  };

  return (
    <div className="space-y-2">
      <Label>Ürünleri Seç</Label>
      <Popover open={open} onOpenChange={setOpen}>
        <PopoverTrigger asChild>
          <Button
            variant="outline"
            role="combobox"
            aria-expanded={open}
            className="w-full justify-between"
          >
            {selectedProductIds.length > 0
              ? `${selectedProductIds.length} ürün seçildi`
              : "Ürün seç..."}
            <ChevronsUpDown className="ml-2 h-4 w-4 shrink-0 opacity-50" />
          </Button>
        </PopoverTrigger>
        <PopoverContent className="w-[--radix-popover-trigger-width] p-0">
          <Command>
            <CommandInput
              placeholder="Ürün ara..."
              value={searchValue}
              onValueChange={setSearchValue}
            />
            <CommandList>
              <CommandEmpty>Ürün bulunamadı.</CommandEmpty>
              <CommandGroup>
                {allProducts.map((product) => (
                  <CommandItem
                    key={product.id}
                    value={product.name}
                    onSelect={() => handleSelect(product.id)}
                  >
                    <Check
                      className={cn(
                        "mr-2 h-4 w-4",
                        selectedProductIds.includes(product.id) ? "opacity-100" : "opacity-0"
                      )}
                    />
                    {product.name}
                  </CommandItem>
                ))}
              </CommandGroup>
            </CommandList>
          </Command>
        </PopoverContent>
      </Popover>
      <div className="flex flex-wrap gap-2 mt-2">
        {selectedProductIds.map((id) => (
          <Badge key={id} variant="secondary" className="flex items-center gap-1">
            {getProductName(id)}
            <Button
              type="button"
              size="icon"
              variant="ghost"
              className="h-4 w-4"
              onClick={() => handleSelect(id)}
            >
              <X className="h-3 w-3" />
            </Button>
          </Badge>
        ))}
      </div>
    </div>
  );
};

export default ProductSelect;
