import { useEffect, useState } from "react"
import { Tabs, TabsList, TabsTrigger, TabsContent } from "@/components/ui/tabs"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { db } from "@/lib/firebase"
import { collection, getDocs, updateDoc, doc, deleteDoc } from "firebase/firestore"
import type { Category } from "@/types"
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogFooter } from "@/components/ui/dialog"
import { useToast } from "@/components/ui/use-toast"
import { Checkbox } from "@/components/ui/checkbox"

interface ProductRequest {
  id: string
  productName: string
  brand: string
  category: string
  buyLink: string
  description: string
  status: string
  createdAt?: any
}

interface ProductRequestsTabProps {
  categories: Category[]
}

export default function ProductRequestsTab({ categories }: ProductRequestsTabProps) {
  const [requests, setRequests] = useState<ProductRequest[]>([])
  const [loading, setLoading] = useState(true)
  const [activeTab, setActiveTab] = useState("pending")
  const [selectedRequest, setSelectedRequest] = useState<ProductRequest | null>(null)
  const [modalOpen, setModalOpen] = useState(false)
  const { toast } = useToast()
  const [selectedIds, setSelectedIds] = useState<string[]>([])

  useEffect(() => {
    fetchRequests()
  }, [])

  const fetchRequests = async () => {
    setLoading(true)
    const snapshot = await getDocs(collection(db, "productRequests"))
    const data: ProductRequest[] = []
    snapshot.forEach(docSnap => {
      data.push({ id: docSnap.id, ...docSnap.data() } as ProductRequest)
    })
    setRequests(data)
    setLoading(false)
  }

  const pendingRequests = requests.filter(r => r.status === "pending")
  const acceptedRequests = requests.filter(r => r.status === "accepted")

  const getCategoryName = (categoryId: string) => {
    return categories.find(cat => cat.id === categoryId)?.name || categoryId
  }

  const handleReview = (req: ProductRequest) => {
    setSelectedRequest(req)
    setModalOpen(true)
  }

  const handleDelete = async (id: string) => {
    await deleteDoc(doc(db, "productRequests", id))
    setRequests((prev) => prev.filter((r) => r.id !== id))
    setModalOpen(false)
    toast({ title: "Talep silindi", variant: "destructive" })
  }

  const handleAccept = async (id: string) => {
    await updateDoc(doc(db, "productRequests", id), { status: "accepted" })
    setRequests((prev) => prev.map(r => r.id === id ? { ...r, status: "accepted" } : r))
    setModalOpen(false)
    toast({ title: "Talep kabul edildi" })
  }

  const handleSelect = (id: string, checked: boolean) => {
    setSelectedIds((prev) => checked ? [...prev, id] : prev.filter(x => x !== id))
  }

  const handleSelectAll = (ids: string[], checked: boolean) => {
    setSelectedIds(checked ? ids : [])
  }

  const handleBulkDelete = async () => {
    await Promise.all(selectedIds.map(id => deleteDoc(doc(db, "productRequests", id))))
    setRequests((prev) => prev.filter(r => !selectedIds.includes(r.id)))
    setSelectedIds([])
    toast({ title: "Seçili talepler silindi", variant: "destructive" })
  }

  return (
    <div>
      <Tabs value={activeTab} onValueChange={setActiveTab} className="mb-4">
        <TabsList>
          <TabsTrigger value="pending">Bekleyenler</TabsTrigger>
          <TabsTrigger value="accepted">Kabul Edilenler</TabsTrigger>
        </TabsList>
      </Tabs>
      {loading ? (
        <div className="text-center py-8 text-gray-500">Yükleniyor...</div>
      ) : (
        <div>
          {activeTab === "pending" && (
            <div className="space-y-4">
              {pendingRequests.length > 0 && (
                <div className="flex items-center gap-2 mb-2">
                  <Checkbox
                    checked={selectedIds.length === pendingRequests.length}
                    onCheckedChange={checked => handleSelectAll(pendingRequests.map(r => r.id), !!checked)}
                    id="select-all-pending"
                  />
                  <label htmlFor="select-all-pending" className="text-xs">Tümünü Seç</label>
                  {selectedIds.length > 0 && (
                    <Button size="sm" variant="destructive" onClick={handleBulkDelete}>Seçilileri Sil</Button>
                  )}
                </div>
              )}
              {pendingRequests.length === 0 ? (
                <div className="text-center text-gray-400">Bekleyen talep yok.</div>
              ) : (
                pendingRequests.map(req => (
                  <Card key={req.id}>
                    <CardHeader className="flex flex-row items-center gap-2">
                      <Checkbox checked={selectedIds.includes(req.id)} onCheckedChange={checked => handleSelect(req.id, !!checked)} />
                      <CardTitle>{req.productName} <span className="text-xs text-gray-500">({req.brand})</span></CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="text-sm text-gray-700 mb-2">Kategori: {getCategoryName(req.category)}</div>
                      <div className="text-xs text-gray-500 mb-2">Açıklama: {req.description || "-"}</div>
                      <div className="flex gap-2">
                        <Button size="sm" variant="outline" onClick={() => handleReview(req)}>İncele</Button>
                        <Button size="sm" variant="destructive" onClick={() => handleDelete(req.id)}>Reddet</Button>
                      </div>
                    </CardContent>
                  </Card>
                ))
              )}
            </div>
          )}
          {activeTab === "accepted" && (
            <div className="space-y-4">
              {acceptedRequests.length > 0 && (
                <div className="flex items-center gap-2 mb-2">
                  <Checkbox
                    checked={selectedIds.length === acceptedRequests.length}
                    onCheckedChange={checked => handleSelectAll(acceptedRequests.map(r => r.id), !!checked)}
                    id="select-all-accepted"
                  />
                  <label htmlFor="select-all-accepted" className="text-xs">Tümünü Seç</label>
                  {selectedIds.length > 0 && (
                    <Button size="sm" variant="destructive" onClick={handleBulkDelete}>Seçilileri Sil</Button>
                  )}
                </div>
              )}
              {acceptedRequests.length === 0 ? (
                <div className="text-center text-gray-400">Kabul edilen talep yok.</div>
              ) : (
                acceptedRequests.map(req => (
                  <Card key={req.id}>
                    <CardHeader className="flex flex-row items-center gap-2">
                      <Checkbox checked={selectedIds.includes(req.id)} onCheckedChange={checked => handleSelect(req.id, !!checked)} />
                      <CardTitle>{req.productName} <span className="text-xs text-gray-500">({req.brand})</span></CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="text-sm text-gray-700 mb-2">Kategori: {getCategoryName(req.category)}</div>
                      <div className="text-xs text-gray-500 mb-2">Açıklama: {req.description || "-"}</div>
                      <div className="flex gap-2">
                        <Button size="sm" variant="outline" onClick={() => handleReview(req)}>Gözden Geçir</Button>
                        <Button size="sm" variant="destructive" onClick={() => handleDelete(req.id)}>Sil</Button>
                      </div>
                    </CardContent>
                  </Card>
                ))
              )}
            </div>
          )}
        </div>
      )}
      <Dialog open={modalOpen} onOpenChange={setModalOpen}>
        <DialogContent className="max-w-md">
          <DialogHeader>
            <DialogTitle>Talep Detayı</DialogTitle>
          </DialogHeader>
          {selectedRequest && (
            <div className="space-y-2">
              <div><b>Ürün Adı:</b> {selectedRequest.productName}</div>
              <div><b>Marka:</b> {selectedRequest.brand}</div>
              <div><b>Kategori:</b> {getCategoryName(selectedRequest.category)}</div>
              <div><b>Satın Alma Linki:</b> {selectedRequest.buyLink ? <a href={selectedRequest.buyLink} target="_blank" rel="noopener noreferrer" className="text-blue-600 underline">{selectedRequest.buyLink}</a> : "-"}</div>
              <div><b>Açıklama:</b> {selectedRequest.description || "-"}</div>
            </div>
          )}
          <DialogFooter className="flex flex-row gap-2 justify-end">
            <Button variant="destructive" onClick={() => selectedRequest && handleDelete(selectedRequest.id)}>Reddet</Button>
            <Button onClick={() => selectedRequest && handleAccept(selectedRequest.id)} disabled={selectedRequest?.status === "accepted"}>Kabul Et</Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  )
} 