"use client"

import { useState, useEffect } from "react"
import { Search, Edit, Trash2, Plus, Star } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import type { Brand } from "@/types"
import Image from "next/image"
import { deleteBrand, loadProducts, saveBrand } from "@/lib/firebase"
import { Badge } from "@/components/ui/badge"
import { Checkbox } from "@/components/ui/checkbox"
import BrandModal from "./BrandModal"

interface BrandsTabProps {
  brands: Brand[]
  loading: boolean
  onEdit: (brand: Brand) => void
  onDelete: (id: string) => void
  onAdd: () => void
  onRefresh: () => void
}

export default function BrandsTab({ brands, loading, onEdit, onDelete, onAdd, onRefresh }: BrandsTabProps) {
  const [searchQuery, setSearchQuery] = useState("")
  const [deleting, setDeleting] = useState<string | null>(null)
  const [selectedIds, setSelectedIds] = useState<string[]>([])
  const [showBrandModal, setShowBrandModal] = useState<Brand | null>(null)
  const [products, setProducts] = useState<any[]>([])
  const [syncing, setSyncing] = useState(false)

  // Ürünleri yükle (ilk açılışta bir kez)
  useEffect(() => {
    loadProducts().then(setProducts)
  }, [])

  const filteredBrands = brands.filter((brand) => brand.name.toLowerCase().includes(searchQuery.toLowerCase()))

  // Sync butonu: ürünlerde olup markalarda olmayan markaları ekle
  const handleSyncBrands = async () => {
    setSyncing(true)
    try {
      // Ürünlerdeki tüm marka adlarını topla
      const productBrands = Array.from(new Set(products.map(p => typeof p.brand === 'string' ? p.brand.trim() : (p.brand?.name?.trim() || "")).filter(Boolean)))
      // Markalarda olmayanları bul
      const existingBrandNames = brands.map(b => b.name.trim().toLowerCase())
      const missingBrands = productBrands.filter(pb => !existingBrandNames.includes(pb.trim().toLowerCase()))
      // Her eksik marka için yeni bir brand oluştur
      for (const name of missingBrands) {
        await saveBrand({ name, logoUrl: "", countryCode: "TR", stars: 3, likes: 0, dislikes: 0, bio: [] })
      }
      onRefresh()
      alert(`${missingBrands.length} yeni marka eklendi!`)
    } catch (e) {
      alert("Sync sırasında hata oluştu.")
    } finally {
      setSyncing(false)
    }
  }

  const handleDelete = async (id: string) => {
    if (!confirm("Bu markayı silmek istediğinizden emin misiniz?")) return

    try {
      setDeleting(id)
      await deleteBrand(id)
      onDelete(id)
      onRefresh()
    } catch (error) {
      console.error("Marka silinirken hata:", error)
      alert("Marka silinirken bir hata oluştu.")
    } finally {
      setDeleting(null)
    }
  }

  const handleSelect = (id: string) => {
    setSelectedIds((prev) => prev.includes(id) ? prev.filter((sid) => sid !== id) : [...prev, id])
  }

  const handleSelectAll = () => {
    if (selectedIds.length === filteredBrands.length) {
      setSelectedIds([])
    } else {
      setSelectedIds(filteredBrands.map((b) => b.id))
    }
  }

  const handleBulkDelete = async () => {
    if (!selectedIds.length) return
    if (!confirm(`${selectedIds.length} markayı silmek istediğinize emin misiniz?`)) return
    try {
      await Promise.all(selectedIds.map(async (id) => {
        await deleteBrand(id)
        onDelete(id)
      }))
      setSelectedIds([])
      onRefresh()
    } catch (error) {
      alert("Toplu silme sırasında hata oluştu.")
    }
  }

  if (loading) {
    return (
      <Card>
        <CardContent className="flex items-center justify-center py-20">
          <div className="text-center">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto mb-4"></div>
            <p className="text-gray-600">Markalar yükleniyor...</p>
          </div>
        </CardContent>
      </Card>
    )
  }

  return (
    <div className="space-y-6">
      {/* Controls */}
      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <CardTitle>Markalar ({filteredBrands.length})</CardTitle>
            <div className="flex gap-2">
              <Button onClick={handleSyncBrands} disabled={syncing} variant="secondary">
                {syncing ? "Sync..." : "Sync"}
              </Button>
              <Button variant="destructive" onClick={handleBulkDelete} disabled={!selectedIds.length}>
                Seçilenleri Sil
              </Button>
            <Button onClick={onAdd}>
              <Plus className="h-4 w-4 mr-2" />
              Yeni Marka
            </Button>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          <div className="relative flex items-center gap-2 mb-2">
            <Checkbox
              checked={selectedIds.length === filteredBrands.length && filteredBrands.length > 0
                ? true
                : selectedIds.length > 0
                ? 'indeterminate'
                : false}
              onCheckedChange={handleSelectAll}
            />
            <span className="text-xs text-gray-500 select-none">Tümünü Seç</span>
            <Search className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
            <Input
              placeholder="Marka adı ara..."
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              className="pl-10"
            />
          </div>
        </CardContent>
      </Card>

      {/* Brands Grid */}
      <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-4 xl:grid-cols-4 gap-3 md:gap-6">
        {filteredBrands.map((brand) => {
          const likes = brand.likes || 0
          const dislikes = brand.dislikes || 0
          const stars = brand.stars || 0
          const totalVotes = likes + dislikes
          const likeRatio = totalVotes > 0 ? (likes / totalVotes) * 100 : 0

          return (
            <Card key={brand.id} className="hover:shadow-lg transition-shadow cursor-pointer" onClick={() => setShowBrandModal(brand)}>
              <CardContent className="p-6">
                <div className="flex items-start justify-between mb-4">
                  <div className="flex items-center gap-3">
                    <Checkbox checked={selectedIds.includes(brand.id)} onCheckedChange={() => handleSelect(brand.id)} />
                    <div className="w-12 h-12 bg-gray-100 rounded-lg flex items-center justify-center overflow-hidden">
                      <Image
                        src={brand.logoUrl || "/placeholder-logo.png"}
                        alt={brand.name}
                        width={48}
                        height={48}
                        className="object-contain"
                        onError={(e) => {
                          const target = e.target as HTMLImageElement
                          target.src = "/placeholder-logo.png"
                        }}
                      />
                    </div>
                    <div>
                      <h3 className="font-semibold text-lg">{brand.name}</h3>
                      <div className="flex items-center gap-2">
                        <div className="flex items-center gap-1">
                          {[...Array(5)].map((_, i) => (
                            <Star
                              key={i}
                              className={`h-4 w-4 ${
                                i < Math.floor(stars) 
                                  ? "fill-yellow-400 text-yellow-400" 
                                  : i === Math.floor(stars) && stars % 1 > 0
                                  ? "fill-yellow-400 text-yellow-400 opacity-60"
                                  : "text-gray-300"
                              }`}
                            />
                          ))}
                          <span className="text-xs text-gray-600 ml-1">({stars.toFixed(1)})</span>
                        </div>
                        <span className="text-xs bg-gray-100 px-2 py-1 rounded">
                          {brand.countryCode || 'XX'}
                        </span>
                        {brand.countryCode === 'TR' && (
                          <Badge className="bg-red-100 text-red-800 text-xs">
                            🇹🇷 Yerli & Milli
                          </Badge>
                        )}
                      </div>
                    </div>
                  </div>

                  <div className="flex items-center gap-1">
                    <Button size="sm" variant="outline" onClick={() => onEdit(brand)}>
                      <Edit className="h-4 w-4" />
                    </Button>
                    <Button
                      size="sm"
                      variant="outline"
                      className="text-red-600 hover:text-red-700"
                      onClick={() => handleDelete(brand.id)}
                      disabled={deleting === brand.id}
                    >
                      <Trash2 className="h-4 w-4" />
                    </Button>
                  </div>
                </div>

                {/* Bio */}
                {brand.bio && brand.bio.length > 0 && (
                  <div className="mb-4">
                    <ul className="text-sm text-gray-600 space-y-2">
                      {brand.bio.slice(0, 3).map((item, index) => (
                        <li key={index} className="flex items-start gap-2">
                          <span className="text-blue-600 mt-1">•</span>
                          <div className="flex-1">
                            <span>{item.content}</span>
                            {item.reference && (
                              <div className="mt-1">
                                <a
                                  href={item.reference}
                                  target="_blank"
                                  rel="noopener noreferrer"
                                  className="text-xs text-blue-600 hover:text-blue-800 underline"
                                  title="Detaylar için tıklayınız"
                                >
                                  📎 Kaynak
                                </a>
                              </div>
                            )}
                          </div>
                        </li>
                      ))}
                    </ul>
                  </div>
                )}

                {/* Stats */}
                <div className="flex items-center justify-between text-sm">
                  <div className="flex items-center gap-4">
                    <span className="flex items-center gap-1">👍 {likes}</span>
                    <span className="flex items-center gap-1">👎 {dislikes}</span>
                  </div>
                  <div className="text-right">
                    <div className="font-medium">{likeRatio.toFixed(1)}%</div>
                    <div className="text-xs text-gray-500">beğeni oranı</div>
                  </div>
                </div>

                {/* Like Ratio Bar */}
                <div className="mt-3">
                  <div className="w-full bg-gray-200 rounded-full h-2">
                    <div
                      className={`h-2 rounded-full transition-all duration-500 ${
                        likeRatio >= 80 ? "bg-green-500" : 
                        likeRatio >= 60 ? "bg-yellow-500" : 
                        likeRatio >= 40 ? "bg-orange-500" : 
                        likeRatio >= 20 ? "bg-red-400" : "bg-red-600"
                      }`}
                      style={{ width: `${likeRatio}%` }}
                    />
                  </div>
                  <div className="text-xs text-gray-500 text-center mt-1">
                    {likeRatio === 0 && "Henüz oylama yok"}
                    {likeRatio > 0 && likeRatio < 20 && "Çok düşük beğeni"}
                    {likeRatio >= 20 && likeRatio < 40 && "Düşük beğeni"}
                    {likeRatio >= 40 && likeRatio < 60 && "Orta beğeni"}
                    {likeRatio >= 60 && likeRatio < 80 && "İyi beğeni"}
                    {likeRatio >= 80 && "Çok iyi beğeni"}
                  </div>
                </div>
              </CardContent>
            </Card>
          )
        })}
      </div>

      {filteredBrands.length === 0 && (
        <Card>
          <CardContent className="text-center py-12">
            <p className="text-gray-500">Arama kriterine uygun marka bulunamadı.</p>
          </CardContent>
        </Card>
      )}

      {/* Brand Modal */}
      {showBrandModal && (
        <BrandModal
          brand={showBrandModal}
          onClose={() => setShowBrandModal(null)}
          onSave={() => setShowBrandModal(null)}
          products={products}
          facilities={[]}
        />
      )}
    </div>
  )
}
