"use client"

import type React from "react"
import { useState, useEffect } from "react"
import { ThumbsUp, ThumbsDown, Eye, Copy, Check } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { useToast } from "@/components/ui/use-toast"
import type { Product } from "@/types"
import Image from "next/image"
import Link from "next/link"
import {
  getTotalSafetyScore,
  getAISafetyScore,
  getCleanContentScore,
  getCertificateScore,
  getCertificates,
  getScoreColor,
  getScoreBadgeColor
} from "@/lib/score-calculator"
import { handleVote, getLocalStorageVote } from "@/lib/firebase"

interface ProductCardProps {
  product: Product
  onRefresh?: () => Promise<void>
  categories?: Array<{ id: string; name: string }>
  brands?: any[]
}

export function ProductCard({ product, onRefresh, categories = [], brands = [] }: ProductCardProps) {
  const [userVote, setUserVote] = useState<'like' | 'dislike' | null>(null)
  const [voting, setVoting] = useState(false)
  const [copied, setCopied] = useState(false)
  const { toast } = useToast()

  useEffect(() => {
    const savedVote = getLocalStorageVote('product', product.id)
    setUserVote(savedVote)
  }, [product.id])

  const handleVoteClick = async (voteType: 'like' | 'dislike', e: React.MouseEvent) => {
    e.preventDefault();
    e.stopPropagation();

    if (voting) return

    setVoting(true)

    try {
      const result = await handleVote('product', product.id, voteType)

      if (result.success) {
        setUserVote(result.currentVote)

        if (onRefresh) {
          await onRefresh()
        }

        toast({
          title: "Oylama",
          description: result.message,
          duration: 2000,
        })
      } else {
        toast({
          title: "Hata",
          description: result.message,
          variant: "destructive",
          duration: 3000,
        })
      }
    } catch (error) {
      console.error('Oylama hatası:', error)
      toast({
        title: "Hata",
        description: "Oylama kaydedilirken bir hata oluştu",
        variant: "destructive",
        duration: 3000,
      })
    } finally {
      setVoting(false)
    }
  }

  // Score calculations
  const aiSafetyScore = getAISafetyScore(product)
  const cleanContentScore = getCleanContentScore(product)
  const certificateScore = getCertificateScore(product)
  const totalVotes = product.likes + product.dislikes
  const popularityScore = totalVotes === 0 ? 0 : Math.round((product.likes / totalVotes) * 100)
  const avg = (aiSafetyScore + cleanContentScore + certificateScore) / 3
  const computedTotalSafetyScore = Math.round(avg * 0.75 + popularityScore * 0.25)

  const getScoreBarColor = (score: number) => {
    return score >= 80 ? 'bg-green-500' : score >= 60 ? 'bg-yellow-500' : 'bg-red-500'
  }

  return (
    <Link href={`/product/${product.id}`} className="block h-full group">
      <div className="h-full flex flex-col bg-white rounded-xl border border-gray-200 shadow-sm hover:shadow-lg transition-all duration-300 hover:-translate-y-1 overflow-hidden">
        {/* Image Container */}
        <div className="relative aspect-square bg-gray-50">
          <Image
            src={product.image || "/placeholder.svg"}
            alt={product.name}
            fill
            className="object-contain p-2 sm:p-3 group-hover:scale-105 transition-transform duration-300"
            sizes="(max-width: 640px) 50vw, (max-width: 768px) 33vw, (max-width: 1024px) 25vw, 20vw"
          />

          {/* Yerli Badge */}
          {(product.yerliMarka || (typeof product.brand === 'object' && product.brand.countryCode === 'TR')) && (
            <div className="absolute top-1 right-1 sm:top-2 sm:right-2">
              <Badge className="bg-red-100 text-red-800 text-[10px] sm:text-xs font-medium px-1 py-0.5">
                🇹🇷
              </Badge>
            </div>
          )}
        </div>

        {/* Content */}
        <div className="p-2 sm:p-3 flex flex-col flex-grow space-y-2">
          {/* Product Info */}
          <div className="flex-grow">
            <h3 className="font-semibold text-xs sm:text-sm mb-1 group-hover:text-blue-600 transition-colors leading-tight line-clamp-2">
              {product.name}
            </h3>
            <p className="text-[10px] sm:text-xs text-gray-500 truncate">
              {typeof product.brand === 'string' ? product.brand : (product.brand ? product.brand.name : '')}
            </p>
          </div>

          {/* Score Section */}
          <div className="space-y-2 pt-2 border-t border-gray-100">
            <div className="flex items-center justify-between">
              <span className="text-[10px] sm:text-xs font-medium text-gray-700">Değerlendirme</span>
              <Badge className={`${getScoreBadgeColor(computedTotalSafetyScore)} text-[10px] px-1 py-0.5 rounded-full`}>
                {computedTotalSafetyScore}%
              </Badge>
            </div>
            <div className="h-1.5 bg-gray-200 rounded-full overflow-hidden">
              <div
                className={`h-full ${getScoreBarColor(computedTotalSafetyScore)} transition-all duration-500`}
                style={{ width: `${computedTotalSafetyScore}%` }}
              />
            </div>
          </div>

          {/* Action Buttons */}
          <div className="flex items-center justify-between pt-2 gap-1">
            <div className="flex items-center gap-1">
              <Button
                variant={userVote === "like" ? "default" : "outline"}
                size="sm"
                onClick={(e) => handleVoteClick("like", e)}
                disabled={voting}
                className="h-6 sm:h-7 px-1.5 sm:px-2 text-[10px] sm:text-xs flex items-center gap-0.5"
              >
                👍 <span className="hidden sm:inline">{product.likes}</span>
                <span className="sm:hidden">{product.likes > 99 ? '99+' : product.likes}</span>
              </Button>
              <Button
                variant={userVote === "dislike" ? "destructive" : "outline"}
                size="sm"
                onClick={(e) => handleVoteClick("dislike", e)}
                disabled={voting}
                className="h-6 sm:h-7 px-1.5 sm:px-2 text-[10px] sm:text-xs flex items-center gap-0.5"
              >
                👎 <span className="hidden sm:inline">{product.dislikes}</span>
                <span className="sm:hidden">{product.dislikes > 99 ? '99+' : product.dislikes}</span>
              </Button>
            </div>

            <div className="flex items-center gap-0.5 text-blue-600 text-[10px] sm:text-xs font-medium">
              <Eye className="h-3 w-3 sm:h-4 sm:w-4" />
              <span className="hidden sm:inline">Detay</span>
            </div>
          </div>
        </div>
      </div>
    </Link>
  )
}
