"use client"

import { useState } from "react"
import { Home, Filter, Heart, Package, Plus, ChevronDown, Award } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Sheet, SheetContent, SheetHeader, SheetTitle, SheetTrigger } from "@/components/ui/sheet"
import { Checkbox } from "@/components/ui/checkbox"
import { Label } from "@/components/ui/label"
import { Separator } from "@/components/ui/separator"
import { Badge } from "@/components/ui/badge"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Collapsible, CollapsibleContent, CollapsibleTrigger } from "@/components/ui/collapsible"
import Link from "next/link"
import type { Product, Category } from "@/types"
import { Input } from "@/components/ui/input"
import { cn } from "@/lib/utils"

interface MobileBottomBarProps {
  onSearch: (query: string) => void
  onCategoryChange: (category: string) => void
  filters: any
  onFiltersChange: (filters: any) => void
  products?: Product[]
  brands?: any[]
  facilities?: any[]
  categories: Category[]
  type?: 'product' | 'brand' | 'facility'
  showFilterButton?: boolean
}

const MobileBottomBar: React.FC<MobileBottomBarProps> = ({
  onSearch,
  onCategoryChange,
  filters,
  onFiltersChange,
  products = [],
  brands = [],
  facilities = [],
  categories: propCategories,
  type = 'product',
  showFilterButton = true,
}) => {
  const [isFilterOpen, setIsFilterOpen] = useState(false)
  const [localFilters, setLocalFilters] = useState(filters)
  const [openSections, setOpenSections] = useState<string[]>([])

  const allSubCategories = propCategories.find(cat => cat.id === localFilters.category)?.subCategories.map(sc => sc.name) || []

  const handleFilterChange = (key: string, value: any) => {
    const newFilters = { ...localFilters, [key]: value }
    setLocalFilters(newFilters)
    onFiltersChange(newFilters)
  }

  const handleSubCategoryChange = (subCategory: string, checked: boolean) => {
    const newSubCategories = checked
      ? [...localFilters.subCategories, subCategory]
      : localFilters.subCategories.filter(cat => cat !== subCategory)
    handleFilterChange('subCategories', newSubCategories)
  }

  const handleBrandChange = (brand: string, checked: boolean) => {
    const newBrands = checked
      ? [...localFilters.brands, brand]
      : localFilters.brands.filter(b => b !== brand)
    handleFilterChange('brands', newBrands)
  }

  const handleContentChange = (content: string, checked: boolean) => {
    const newContents = checked
      ? [...localFilters.contents, content]
      : localFilters.contents.filter(c => c !== content)
    handleFilterChange('contents', newContents)
  }

  const clearAllFilters = () => {
    const clearedFilters = {
      ...localFilters,
      category: "*",
      subCategories: [],
      brands: [],
      contents: [],
      rating: { green: false, red: false },
      yerliMarka: false
    }
    setLocalFilters(clearedFilters)
    onFiltersChange(clearedFilters)
  }

  const toggleSection = (section: string) => {
    setOpenSections(prev =>
      prev.includes(section)
        ? prev.filter(s => s !== section)
        : [...prev, section]
    )
  }

  const activeFilterCount = (localFilters.category !== "*" ? 1 : 0) +
    localFilters.subCategories.length +
    localFilters.brands.length +
    localFilters.contents.length +
    (localFilters.rating.green ? 1 : 0) +
    (localFilters.rating.red ? 1 : 0) +
    (localFilters.yerliMarka ? 1 : 0)

  return (
    <>
      {/* Bottom Navigation Bar */}
      <div className="fixed bottom-0 left-0 right-0 bg-white/95 backdrop-blur-md border-t border-gray-200 shadow-2xl z-[9999]">
        <div className="flex items-center justify-between h-16 px-2">
          {/* Home */}
          <Link href="/" className="flex flex-col items-center justify-center flex-1 h-full group">
            <Home className="h-5 w-5 text-gray-600 group-hover:text-blue-600 transition-colors mb-1" />
            <span className="text-[10px] text-gray-600 group-hover:text-blue-600 transition-colors">Ana Sayfa</span>
          </Link>

          {/* Filter */}
          {showFilterButton && (
            <Sheet open={isFilterOpen} onOpenChange={setIsFilterOpen}>
              <SheetTrigger asChild>
                <div className="flex flex-col items-center justify-center flex-1 h-full group relative">
                  <Filter className="h-5 w-5 text-gray-600 group-hover:text-blue-600 transition-colors mb-1" />
                  <span className="text-[10px] text-gray-600 group-hover:text-blue-600 transition-colors">Filtrele</span>
                  {activeFilterCount > 0 && (
                    <Badge className="absolute -top-1 -right-1 h-5 w-5 p-0 text-[10px] flex items-center justify-center bg-blue-600 text-white rounded-full">
                      {activeFilterCount}
                    </Badge>
                  )}
                </div>
              </SheetTrigger>
              <SheetContent
                side="bottom"
                className="h-[85vh] overflow-y-auto rounded-t-xl"
                onPointerDownOutside={() => setIsFilterOpen(false)}
                onInteractOutside={() => setIsFilterOpen(false)}
              >
                <SheetHeader className="pb-4">
                  <SheetTitle className="text-left flex items-center gap-2">
                    <Filter className="h-5 w-5 text-blue-600" />
                    Filtreler
                  </SheetTitle>
                </SheetHeader>

                <div className="space-y-6 pb-24">
                  {type === 'brand' ? (
                    <>
                      <Collapsible open={openSections.includes('brand-sort')} onOpenChange={() => toggleSection('brand-sort')} className="space-y-3">
                        <CollapsibleTrigger className="flex justify-between items-center w-full font-medium text-gray-900 p-3 bg-gray-50 rounded-lg">
                          <h3>Sırala</h3>
                          <ChevronDown className={cn("h-4 w-4 transition-transform duration-200", openSections.includes('brand-sort') && "rotate-180")} />
                        </CollapsibleTrigger>
                        <CollapsibleContent>
                          <Select
                            value={localFilters.sortBy || 'stars'}
                            onValueChange={(value) => handleFilterChange('sortBy', value)}
                          >
                            <SelectTrigger className="w-full">
                              <SelectValue placeholder="Sıralama seçin" />
                            </SelectTrigger>
                            <SelectContent>
                              <SelectItem value="stars">Puan (Yıldız)</SelectItem>
                              <SelectItem value="like">Beğeni (Like)</SelectItem>
                              <SelectItem value="dislike">Beğenmeme (Dislike)</SelectItem>
                              <SelectItem value="likeRatio">Beğeni Oranı</SelectItem>
                              <SelectItem value="country">Ülke</SelectItem>
                              <SelectItem value="yerli">Yerli & Milli</SelectItem>
                            </SelectContent>
                          </Select>
                        </CollapsibleContent>
                      </Collapsible>

                      <Separator />

                      <Collapsible open={openSections.includes('brand-country')} onOpenChange={() => toggleSection('brand-country')} className="space-y-3">
                        <CollapsibleTrigger className="flex justify-between items-center w-full font-medium text-gray-900 p-3 bg-gray-50 rounded-lg">
                          <h3>Ülke</h3>
                          <ChevronDown className={cn("h-4 w-4 transition-transform duration-200", openSections.includes('brand-country') && "rotate-180")} />
                        </CollapsibleTrigger>
                        <CollapsibleContent>
                          <div className="space-y-3 max-h-48 overflow-y-auto">
                            {Array.from(new Set((brands || []).map(function(b: any): string { return b.countryCode || 'Diğer' }))).map(function(country: string): JSX.Element {
                              return (
                                <div key={country} className="flex items-center space-x-3 p-2 hover:bg-gray-50 rounded">
                                  <Checkbox
                                    id={`country-${country}`}
                                    checked={localFilters.countries?.includes(country) || false}
                                    onCheckedChange={(checked) => {
                                      const newCountries = checked
                                        ? [...(localFilters.countries || []), country]
                                        : (localFilters.countries || []).filter(function(c: any): boolean { return c !== country })
                                      handleFilterChange('countries', newCountries)
                                    }}
                                  />
                                  <Label htmlFor={`country-${country}`} className="text-sm font-medium">
                                    {country}
                                  </Label>
                                </div>
                              )
                            })}
                          </div>
                        </CollapsibleContent>
                      </Collapsible>
                    </>
                  ) : type === 'facility' ? (
                    <>
                      {/* Facility filters content */}
                    </>
                  ) : (
                    <>
                      {/* Product Filters */}
                      <Collapsible open={openSections.includes('product-category')} onOpenChange={() => toggleSection('product-category')} className="space-y-3">
                        <CollapsibleTrigger className="flex justify-between items-center w-full font-medium text-gray-900 p-3 bg-blue-50 rounded-lg">
                          <h3 className="flex items-center gap-2">
                            <span className="bg-blue-600 text-white rounded-full w-6 h-6 flex items-center justify-center text-sm font-bold">1</span>
                            Kategori
                          </h3>
                          <ChevronDown className={cn("h-4 w-4 transition-transform duration-200", openSections.includes('product-category') && "rotate-180")} />
                        </CollapsibleTrigger>
                        <CollapsibleContent>
                          <Select
                            value={localFilters.category}
                            onValueChange={(value) => handleFilterChange('category', value)}
                          >
                            <SelectTrigger className="w-full">
                              <SelectValue placeholder="Ana kategori seçin" />
                            </SelectTrigger>
                            <SelectContent>
                              <SelectItem key="*" value="*">Tümü</SelectItem>
                              {propCategories.map((category) => (
                                <SelectItem key={category.id} value={category.id}>
                                  {category.name}
                                </SelectItem>
                              ))}
                            </SelectContent>
                          </Select>
                        </CollapsibleContent>
                      </Collapsible>

                      <Separator />

                      {/* Alt Kategori */}
                      <Collapsible open={openSections.includes('product-subcategory')} onOpenChange={() => toggleSection('product-subcategory')} className="space-y-3">
                        <CollapsibleTrigger className="flex justify-between items-center w-full font-medium text-gray-900 p-3 bg-green-50 rounded-lg">
                          <h3 className="flex items-center gap-2">
                            <span className="bg-green-600 text-white rounded-full w-6 h-6 flex items-center justify-center text-sm font-bold">2</span>
                            Alt Kategori
                          </h3>
                          <ChevronDown className={cn("h-4 w-4 transition-transform duration-200", openSections.includes('product-subcategory') && "rotate-180")} />
                        </CollapsibleTrigger>
                        <CollapsibleContent>
                          <div className="space-y-3 max-h-48 overflow-y-auto">
                            {allSubCategories.map((subCategory) => (
                              <div key={subCategory} className="flex items-center space-x-3 p-2 hover:bg-gray-50 rounded">
                                <Checkbox
                                  id={`sub-${subCategory}`}
                                  checked={localFilters.subCategories.includes(subCategory)}
                                  onCheckedChange={(checked) =>
                                    handleSubCategoryChange(subCategory, checked as boolean)
                                  }
                                />
                                <Label htmlFor={`sub-${subCategory}`} className="text-sm font-medium">
                                  {subCategory}
                                </Label>
                              </div>
                            ))}
                          </div>
                        </CollapsibleContent>
                      </Collapsible>

                      <Separator />

                      {/* Marka */}
                      <Collapsible open={openSections.includes('product-brand')} onOpenChange={() => toggleSection('product-brand')} className="space-y-3">
                        <CollapsibleTrigger className="flex justify-between items-center w-full font-medium text-gray-900 p-3 bg-purple-50 rounded-lg">
                          <h3 className="flex items-center gap-2">
                            <span className="bg-purple-600 text-white rounded-full w-6 h-6 flex items-center justify-center text-sm font-bold">3</span>
                            Marka
                          </h3>
                          <ChevronDown className={cn("h-4 w-4 transition-transform duration-200", openSections.includes('product-brand') && "rotate-180")} />
                        </CollapsibleTrigger>
                        <CollapsibleContent>
                          <div className="space-y-3 max-h-48 overflow-y-auto">
                            {brands.map((brand) => (
                              <div key={brand} className="flex items-center space-x-3 p-2 hover:bg-gray-50 rounded">
                                <Checkbox
                                  id={`brand-${brand}`}
                                  checked={localFilters.brands.includes(brand)}
                                  onCheckedChange={(checked) =>
                                    handleBrandChange(brand, checked as boolean)
                                  }
                                />
                                <Label htmlFor={`brand-${brand}`} className="text-sm font-medium">
                                  {brand}
                                </Label>
                              </div>
                            ))}
                          </div>
                        </CollapsibleContent>
                      </Collapsible>

                      <Separator />

                      {/* Yerli Marka */}
                      <Collapsible open={openSections.includes('product-yerli')} onOpenChange={() => toggleSection('product-yerli')} className="space-y-3">
                        <CollapsibleTrigger className="flex justify-between items-center w-full font-medium text-gray-900 p-3 bg-red-50 rounded-lg">
                          <h3 className="flex items-center gap-2">
                            <span className="bg-red-600 text-white rounded-full w-6 h-6 flex items-center justify-center text-sm font-bold">4</span>
                            Yerli & Milli
                          </h3>
                          <ChevronDown className={cn("h-4 w-4 transition-transform duration-200", openSections.includes('product-yerli') && "rotate-180")} />
                        </CollapsibleTrigger>
                        <CollapsibleContent>
                          <div className="p-2">
                            <div className="flex items-center space-x-3 p-3 bg-red-50 rounded-lg">
                              <Checkbox
                                id="yerli-marka"
                                checked={localFilters.yerliMarka}
                                onCheckedChange={(checked) =>
                                  handleFilterChange('yerliMarka', checked as boolean)
                                }
                              />
                              <Label htmlFor="yerli-marka" className="text-sm font-medium">
                                🇹🇷 Sadece Yerli ve Milli Markalar
                              </Label>
                            </div>
                          </div>
                        </CollapsibleContent>
                      </Collapsible>
                    </>
                  )}

                  {/* Clear Filters Button */}
                  {activeFilterCount > 0 && (
                    <div className="pt-4">
                      <Button
                        variant="outline"
                        onClick={clearAllFilters}
                        className="w-full py-3 text-base"
                      >
                        Tüm Filtreleri Temizle ({activeFilterCount})
                      </Button>
                    </div>
                  )}
                </div>

                {/* Apply Button */}
                <div className="fixed bottom-4 left-4 right-4 z-50">
                  <Button
                    className="w-full text-base font-semibold py-4 rounded-xl shadow-lg bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700"
                    onClick={() => setIsFilterOpen(false)}
                  >
                    Filtreleri Uygula
                  </Button>
                </div>
              </SheetContent>
            </Sheet>
          )}

          {/* Brands */}
          <Link href="/marka-puanlamasi" className="flex flex-col items-center justify-center flex-1 h-full group">
            <Package className="h-5 w-5 text-gray-600 group-hover:text-blue-600 transition-colors mb-1" />
            <span className="text-[10px] text-gray-600 group-hover:text-blue-600 transition-colors">Markalar</span>
          </Link>

          {/* Facilities */}
          <Link href="/uretim-tesisleri" className="flex flex-col items-center justify-center flex-1 h-full group">
            <Award className="h-5 w-5 text-gray-600 group-hover:text-blue-600 transition-colors mb-1" />
            <span className="text-[10px] text-gray-600 group-hover:text-blue-600 transition-colors">Tesisler</span>
          </Link>

          {/* Contact */}
          <Link href="/iletisim" className="flex flex-col items-center justify-center flex-1 h-full group">
            <Heart className="h-5 w-5 text-gray-600 group-hover:text-blue-600 transition-colors mb-1" />
            <span className="text-[10px] text-gray-600 group-hover:text-blue-600 transition-colors">İletişim</span>
          </Link>
        </div>
      </div>

      {/* Bottom padding for content */}
      <div className="h-16" />
    </>
  );
}

export default MobileBottomBar
