"use client"

import React, { useState, useRef } from "react"
import { Search, ArrowLeft } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import Link from "next/link"
import type { Category, Product, Brand, Facility } from "@/types"

interface HeaderProps {
  onSearch?: (query: string) => void
  onCategoryChange?: (category: string) => void
  showCategories?: boolean
  categories: Category[]
  products?: Product[]
  brands?: Brand[]
  facilities?: Facility[]
  showSearchBox?: boolean
}

export default function Header({
  onSearch,
  onCategoryChange,
  showCategories = true,
  categories = [],
  products = [],
  brands = [],
  facilities = [],
  showSearchBox = true
}: HeaderProps) {
  const [searchQuery, setSearchQuery] = useState("")
  const [activeCategory, setActiveCategory] = useState("*")
  const [showResults, setShowResults] = useState(false)
  const [showMobileSearchInput, setShowMobileSearchInput] = useState(false)
  const searchBoxRef = useRef<HTMLDivElement>(null)

  const lowerQuery = searchQuery.toLowerCase()
  const filteredProducts = products?.filter((p) =>
    p.name?.toLowerCase().includes(lowerQuery) ||
    (typeof p.brand === 'string' ? p.brand.toLowerCase().includes(lowerQuery) : p.brand?.name?.toLowerCase().includes(lowerQuery))
  ) || []
  const filteredBrands = brands?.filter((b) => b.name?.toLowerCase().includes(lowerQuery)) || []
  const filteredFacilities = facilities?.filter((f) => f.name?.toLowerCase().includes(lowerQuery)) || []

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault()
    onSearch && onSearch(searchQuery)
    setShowResults(false)
    setShowMobileSearchInput(false)
  }

  const handleCategoryClick = (categoryId: string) => {
    setActiveCategory(categoryId)
    onCategoryChange && onCategoryChange(categoryId)
  }

  React.useEffect(() => {
    function handleClickOutside(event: MouseEvent) {
      if (searchBoxRef.current && !searchBoxRef.current.contains(event.target as Node)) {
        setShowResults(false)
      }
    }
    if (showResults) {
      document.addEventListener("mousedown", handleClickOutside)
      return () => document.removeEventListener("mousedown", handleClickOutside)
    }
  }, [showResults])

  return (
    <>
      {/* Desktop Header */}
      <header className="hidden md:block bg-white shadow-sm sticky top-0 z-50">
        <div className="container mx-auto px-4 py-4">
          <div className="flex items-center gap-6">
            <Link href="/" className="flex-shrink-0">
              <h1 className="text-2xl font-bold text-blue-600">Vitamin & Takviye</h1>
            </Link>

            {showSearchBox && (
              <div className="flex-1 max-w-2xl relative" ref={searchBoxRef}>
                <form onSubmit={handleSearch}>
                  <div className="relative">
                    <Search className="absolute left-4 top-1/2 transform -translate-y-1/2 h-5 w-5 text-gray-400" />
                    <Input
                      type="text"
                      placeholder="Ürün, marka veya tesis ara..."
                      value={searchQuery}
                      onChange={(e) => {
                        setSearchQuery(e.target.value)
                        setShowResults(true)
                      }}
                      className="pl-12 pr-20 py-3 border-gray-300 focus:border-blue-500 focus:ring-blue-500 rounded-full text-base"
                      onFocus={() => setShowResults(true)}
                    />
                    <Button
                      type="submit"
                      size="sm"
                      className="absolute right-2 top-1/2 transform -translate-y-1/2 h-8 px-4 rounded-full bg-blue-600 hover:bg-blue-700"
                    >
                      Ara
                    </Button>
                  </div>

                  {showResults && searchQuery.trim() && (
                    <div className="absolute left-0 right-0 mt-2 bg-white border border-gray-200 rounded-lg shadow-lg z-50 max-h-80 overflow-y-auto">
                      {filteredProducts.length > 0 && (
                        <div className="p-2">
                          <div className="text-xs font-semibold text-gray-500 mb-1">Ürünler</div>
                          {filteredProducts.slice(0, 5).map((p) => (
                            <Link key={p.id} href={`/product/${p.id}`}
                              className="block px-3 py-2 hover:bg-blue-50 rounded transition-colors text-sm text-gray-800"
                              onClick={() => setShowResults(false)}
                            >
                              {p.name} <span className="text-xs text-gray-400">({typeof p.brand === 'string' ? p.brand : p.brand?.name})</span>
                            </Link>
                          ))}
                        </div>
                      )}
                      {filteredProducts.length === 0 && filteredBrands.length === 0 && filteredFacilities.length === 0 && (
                        <div className="p-3 text-sm text-gray-400">Sonuç bulunamadı.</div>
                      )}
                    </div>
                  )}
                </form>
              </div>
            )}
          </div>
        </div>

        {/* Desktop Categories */}
        {showCategories && (
          <div className="bg-blue-50/50 border-t">
            <div className="container mx-auto px-4">
              <nav className="py-2.5">
                <ul className="flex flex-nowrap gap-3 overflow-x-auto scrollbar-thin scrollbar-thumb-blue-200 scrollbar-track-blue-50 py-1">
                  <li>
                    <button
                      onClick={() => handleCategoryClick("*")}
                      className={`px-4 py-2 rounded-full text-sm font-medium transition-all whitespace-nowrap ${
                        activeCategory === "*" ? "bg-blue-600 text-white shadow" : "bg-white text-gray-700 hover:bg-blue-100"
                      }`}
                    >
                      Tümü
                    </button>
                  </li>
                  {categories.slice(0, 8).map((cat) => (
                    <li key={cat.id}>
                      <button
                        onClick={() => handleCategoryClick(cat.id)}
                        className={`px-4 py-2 rounded-full text-sm font-medium transition-all whitespace-nowrap ${
                          activeCategory === cat.id ? "bg-blue-600 text-white shadow" : "bg-white text-gray-700 hover:bg-blue-100"
                        }`}
                      >
                        {cat.name}
                      </button>
                    </li>
                  ))}
                </ul>
              </nav>
            </div>
          </div>
        )}
      </header>

      {/* Mobile Header */}
      <header className="md:hidden bg-white shadow-sm sticky top-0 z-50">
        <div className="flex items-center justify-between px-4 py-3">
          <Link href="/" className="flex-1">
            <h1 className="text-lg font-bold text-blue-600">Vitamin & Takviye</h1>
          </Link>

          {showSearchBox && (
            <Button
              variant="ghost"
              size="sm"
              onClick={() => setShowMobileSearchInput(true)}
              className="p-2"
            >
              <Search className="h-5 w-5 text-gray-600" />
            </Button>
          )}
        </div>

        {/* Mobile Categories */}
        {showCategories && (
          <div className="border-t bg-white px-4 py-2">
            <div className="flex gap-2 overflow-x-auto pb-2 scrollbar-hide">
              <button
                onClick={() => handleCategoryClick("*")}
                className={`px-3 py-1.5 rounded-full text-xs font-medium transition-all whitespace-nowrap flex-shrink-0 ${
                  activeCategory === "*" ? "bg-blue-600 text-white" : "bg-gray-100 text-gray-700"
                }`}
              >
                Tümü
              </button>
              {categories.slice(0, 8).map((cat) => (
                <button
                  key={cat.id}
                  onClick={() => handleCategoryClick(cat.id)}
                  className={`px-3 py-1.5 rounded-full text-xs font-medium transition-all whitespace-nowrap flex-shrink-0 ${
                    activeCategory === cat.id ? "bg-blue-600 text-white" : "bg-gray-100 text-gray-700"
                  }`}
                >
                  {cat.name}
                </button>
              ))}
            </div>
          </div>
        )}
      </header>

      {/* Mobile Search Overlay */}
      {showMobileSearchInput && (
        <div className="md:hidden fixed inset-0 bg-white z-[9999] flex flex-col">
          <div className="flex items-center gap-3 p-4 border-b bg-white">
            <Button
              variant="ghost"
              size="sm"
              onClick={() => setShowMobileSearchInput(false)}
              className="p-2"
            >
              <ArrowLeft className="h-5 w-5" />
            </Button>

            <form onSubmit={handleSearch} className="flex-1 flex gap-2">
              <Input
                type="text"
                placeholder="Ürün, marka ara..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="flex-1 text-base"
                autoFocus
              />
              <Button type="submit" size="sm" className="px-4">
                Ara
              </Button>
            </form>
          </div>

          <div className="flex-1 overflow-y-auto">
            {searchQuery.trim() && (
              <div className="p-4 space-y-4">
                {filteredProducts.length > 0 && (
                  <div>
                    <h3 className="text-sm font-semibold text-gray-500 mb-2">Ürünler</h3>
                    <div className="space-y-1">
                      {filteredProducts.slice(0, 10).map((p) => (
                        <Link
                          key={p.id}
                          href={`/product/${p.id}`}
                          className="block p-3 hover:bg-gray-50 rounded-lg transition-colors"
                          onClick={() => setShowMobileSearchInput(false)}
                        >
                          <div className="font-medium text-sm">{p.name}</div>
                          <div className="text-xs text-gray-500 mt-1">
                            {typeof p.brand === 'string' ? p.brand : p.brand?.name}
                          </div>
                        </Link>
                      ))}
                    </div>
                  </div>
                )}
              </div>
            )}
          </div>
        </div>
      )}
    </>
  )
}
