"use client"

import { useEffect, useState } from "react"
import { Award, ThumbsUp, ThumbsDown, Factory, Users, Package } from "lucide-react"
import { Card, CardContent } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { loadFacilities, loadProducts, loadCategories, saveFacility, loadBrands } from "@/lib/firebase"
import type { Facility, Product, Category } from "@/types"
import Image from "next/image"
import Header from "@/components/Header"
import Footer from "@/components/Footer"
import MobileBottomBar from "@/components/MobileBottomBar"

export default function UretimTesisleriPage() {
  const [facilities, setFacilities] = useState<Facility[]>([])
  const [loading, setLoading] = useState(true)
  const [products, setProducts] = useState<Product[]>([])
  const [categories, setCategories] = useState<Category[]>([])
  const [filters, setFilters] = useState({
    category: "*",
    subCategories: [],
    brands: [],
    contents: [],
    rating: { green: false, red: false },
    search: "",
    yerliMarka: false,
    certificates: [],
    producedBrands: [],
    scoreMin: undefined,
    scoreMax: undefined,
    sortBy: 'score',
  })
  const [visibleCount, setVisibleCount] = useState(30);
  const [openFacility, setOpenFacility] = useState<string | null>(null);
  const [facilityLikes, setFacilityLikes] = useState<Record<string, number>>({});
  const [facilityDislikes, setFacilityDislikes] = useState<Record<string, number>>({});
  // Her tesis için ürün gridinde kaç ürün görünecek? (id: count)
  const [visibleProducts, setVisibleProducts] = useState<Record<string, number>>({});

  const [brands, setBrands] = useState<any[]>([])

  useEffect(() => {
    loadFacilities().then((data) => {
      setFacilities(data)
      setLoading(false)
    })
    loadProducts().then(setProducts)
    loadCategories().then(setCategories)
    loadBrands().then(setBrands)
  }, [])

  

  // Like/dislike işlemleri Firestore'a kaydedilsin ve kullanıcı başına bir oy olsun
  const handleFacilityVote = async (facility: Facility, type: 'like' | 'dislike') => {
    const voteKey = `facility-vote-${facility.id}`;
    if (typeof window !== 'undefined' && localStorage.getItem(voteKey)) return;
    const field = type === 'like' ? 'likes' : 'dislikes';
    // @ts-ignore: likes/dislikes opsiyonel olabilir
    const newValue = (facility[field] || 0) + 1;
    try {
      await saveFacility({ ...facility, [field]: newValue });
      setFacilityLikes(l => ({ ...l, [facility.id]: type === 'like' ? newValue : l[facility.id] || 0 }));
      setFacilityDislikes(d => ({ ...d, [facility.id]: type === 'dislike' ? newValue : d[facility.id] || 0 }));
      if (typeof window !== 'undefined') localStorage.setItem(voteKey, type);
      loadFacilities().then(setFacilities);
    } catch {}
  };

  // Kategorik özetler için hesaplamalar
  const facilityProductCounts = facilities.map(facility => ({
    facility,
    count: 0, // This will be calculated based on products later if needed, but for now, we'll use brandsProduced
    brands: facility.brandsProduced || [],
  }));
  const mostBrandProducingFacilities = [...facilityProductCounts].sort((a, b) => b.brands.length - a.brands.length).slice(0, 5);
  const totalProducedBrands = Array.from(new Set(facilities.flatMap(f => f.brandsProduced || []))).length;

  console.log("DEBUG: facilityProductCounts:", facilityProductCounts);
  
  console.log("DEBUG: mostBrandProducingFacilities:", mostBrandProducingFacilities);
  console.log("DEBUG: totalProducedBrands:", totalProducedBrands);

  return (
    <div className="min-h-screen bg-gray-50 flex flex-col">
      <Header showCategories={false} onSearch={() => {}} onCategoryChange={() => {}} categories={[]} 
        products={products}
        brands={[]}
        facilities={facilities}
      />
      <main className="flex-1 container mx-auto px-4 py-8">
        <div className="text-center mb-8">
          <h1 className="text-3xl font-bold text-gray-900 mb-4 flex items-center justify-center gap-2">
            <Award className="h-8 w-8 text-yellow-500" />
            Üretim Tesisleri
          </h1>
          <p className="text-gray-600">Türkiye ve dünyadaki takviye üretim tesislerini keşfedin</p>
        </div>
        {/* Kategorik özet kutuları */}
        <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-8">
          <div className="bg-blue-50 rounded-lg p-4 flex flex-col items-center">
            <Factory className="h-6 w-6 text-blue-600 mb-1" />
            <div className="text-2xl font-bold text-blue-700">{facilities.length}</div>
            <div className="text-xs text-gray-600">Toplam Tesis</div>
          </div>
          
          <div className="bg-yellow-50 rounded-lg p-4 flex flex-col items-center">
            <Users className="h-6 w-6 text-yellow-600 mb-1" />
            <div className="text-2xl font-bold text-yellow-700">{totalProducedBrands}</div>
            <div className="text-xs text-gray-600">Toplam Üretilen Marka</div>
          </div>
          
        </div>
        {/* En çok marka üreten tesisler kutusu */}
        <div className="bg-pink-50 rounded-lg p-4 flex flex-col items-center mb-8">
          <div className="text-xs text-pink-700 font-semibold mb-1">En Çok Marka Üreten Tesisler</div>
          <ul className="text-xs text-gray-700 list-disc pl-4">
            {mostBrandProducingFacilities.map(f => (
              <li key={f.facility.id}>{f.facility.name} <span className="font-bold">({f.brands.length})</span></li>
            ))}
          </ul>
        </div>
        {loading ? (
          <div className="flex items-center justify-center py-20">
            <div className="text-center">
              <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto mb-4"></div>
              <p className="text-gray-600">Tesisler yükleniyor...</p>
            </div>
          </div>
        ) : (
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6">
            {facilities.slice(0, visibleCount).map((facility) => {
              const facilityBrands = facility.brandsProduced || [];
              const likes = (facility as any).likes || 0;
              const dislikes = (facility as any).dislikes || 0;
              const totalVotes = likes + dislikes;
              const likeRatio = totalVotes > 0 ? Math.round((likes / totalVotes) * 100) : 0;
              return (
                <Card key={facility.id} className="hover:shadow-lg transition-shadow">
                  <CardContent className="p-6">
                    <div className="flex items-center gap-4 mb-4">
                      <div className="w-16 h-16 bg-gray-100 rounded-lg flex items-center justify-center overflow-hidden">
                        <Image
                          src={facility.logoUrl || "/placeholder-logo.png"}
                          alt={facility.name}
                          width={64}
                          height={64}
                          className="object-contain"
                          onError={(e) => {
                            const target = e.target as HTMLImageElement
                            target.src = "/placeholder-logo.png"
                          }}
                        />
                      </div>
                      <div className="flex-1 min-w-0">
                        <h3 className="font-semibold text-lg text-blue-700 truncate">{facility.name}</h3>
                        <div className="flex items-center gap-2 mt-1">
                          <Award className="h-4 w-4 text-yellow-500" />
                          <span className="text-xs text-gray-600">Puan: {facility.score}/100</span>
                        </div>
                        {facility.certificates && (
                          <div className="flex flex-wrap gap-2 mt-2">
                            {Object.entries(facility.certificates).filter(([_, v]) => v).map(([key]) => (
                              <Badge key={key} variant="secondary" className="text-xs">
                                {key.toUpperCase()}
                              </Badge>
                            ))}
                          </div>
                        )}
                      </div>
                    </div>
                    
                    {/* Ürettiği markalar */}
                    {facility.brandsProduced && facility.brandsProduced.length > 0 && (
                      <div className="mb-2 text-xs text-gray-600">
                        <span className="font-semibold">Ürettiği Markalar:</span>
                        <div className="flex flex-wrap gap-2 mt-1">
                          {facility.brandsProduced.map(brandName => {
                            const brand = brands.find(b => b.name === brandName);
                            if (!brand) return null; // Marka bulunamazsa gösterme
                            return (
                              <div key={brand.id || brandName} className="flex items-center gap-1 bg-gray-100 rounded-full px-2 py-1">
                                {brand.logoUrl && (
                                  <Image
                                    src={brand.logoUrl}
                                    alt={brand.name}
                                    width={16}
                                    height={16}
                                    className="object-contain rounded-full"
                                  />
                                )}
                                <span className="text-xs">{brand.name}</span>
                              </div>
                            )
                          })}
                        </div>
                      </div>
                    )}
                    {/* Like/Dislike oranı barı */}
                    <div className="mb-4">
                      <div className="w-full bg-gray-200 rounded-full h-4 relative">
                        <div
                          className="h-4 rounded-full bg-gradient-to-r from-blue-400 to-blue-600 transition-all duration-300"
                          style={{ width: totalVotes === 0 ? '0%' : `${likeRatio}%` }}
                        />
                        <span className="absolute left-1/2 top-1/2 -translate-x-1/2 -translate-y-1/2 text-xs font-semibold text-white drop-shadow">
                          {totalVotes === 0 ? 'N/A' : likeRatio + '% beğeni'}
                        </span>
                      </div>
                      <div className="flex items-center gap-3 mt-1">
                        <button onClick={() => handleFacilityVote(facility, 'like')} className="flex items-center gap-1 text-green-600 hover:text-green-800">
                          <ThumbsUp className="h-4 w-4" /> {likes}
                        </button>
                        <button onClick={() => handleFacilityVote(facility, 'dislike')} className="flex items-center gap-1 text-red-600 hover:text-red-800">
                          <ThumbsDown className="h-4 w-4" /> {dislikes}
                        </button>
                      </div>
                    </div>
                    
                  </CardContent>
                </Card>
              );
            })}
          </div>
        )}
        {visibleCount < facilities.length && (
          <div className="flex justify-center mt-6">
            <button
              className="px-6 py-2 bg-blue-600 text-white rounded hover:bg-blue-700 transition"
              onClick={() => setVisibleCount((prev) => prev + 30)}
            >
              Daha fazla yükle
            </button>
          </div>
        )}
      </main>
      <Footer />
      <MobileBottomBar
        onSearch={(query) => setFilters((prev) => ({ ...prev, search: query }))}
        onCategoryChange={(category) => setFilters((prev) => ({ ...prev, category }))}
        filters={filters}
        onFiltersChange={setFilters}
        products={products}
        categories={categories}
        showFilterButton={false}
      />
    </div>
  )
} 