"use client"

import { useState, useEffect, useMemo } from "react"
import { useIsMobile } from "@/hooks/use-mobile"
import { Plus, BarChart3, Package, Users, TrendingUp } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import ProductsTab from "@/components/admin/ProductsTab"
import BrandsTab from "@/components/admin/BrandsTab"
import StatsTab from "@/components/admin/StatsTab"
import JsonExporter from "../../components/admin/JsonExporter"
import ProductModal from "@/components/admin/ProductModal"
import BrandModal from "@/components/admin/BrandModal"
import MobileBottomBar from "@/components/MobileBottomBar"
import type { Product, Brand, Category, Facility, HomePageSection } from "@/types"
import CategoriesTab from "@/components/admin/CategoriesTab"
import ProductRequestsTab from "@/components/admin/ProductRequestsTab"
import FacilitiesTab from "@/components/admin/FacilitiesTab"
import FacilityModal from "@/components/admin/FacilityModal"
import JsonImporterTab from "@/components/admin/JsonImporterTab"
import HomePageLayoutTab from "@/components/admin/HomePageLayoutTab"
import { CustomWidgetsTab } from "@/components/admin/CustomWidgetsTab"
import ProcessorTab from "@/components/admin/ProcessorTab"
import JsonTab from "@/components/admin/JsonTab"
import { loadProducts, loadBrands, loadCategories, loadFacilities, getHomePageLayout, saveHomePageLayout } from "@/lib/firebase"

export default function AdminPageClient() {
  const isMobile = useIsMobile();
  const [products, setProducts] = useState<Product[]>([])
  const [brands, setBrands] = useState<Brand[]>([])
  const memoizedBrands = useMemo(() => brands, [brands]);
  const [categories, setCategories] = useState<Category[]>([])
  const [facilities, setFacilities] = useState<Facility[]>([])
  const [homePageLayout, setHomePageLayout] = useState<HomePageSection[]>([]);
  const [loading, setLoading] = useState(true)
  const [selectedProduct, setSelectedProduct] = useState<Product | null>(null)
  const [selectedBrand, setSelectedBrand] = useState<Brand | null>(null)
  const [selectedFacility, setSelectedFacility] = useState<Facility | null>(null)
  const [showProductModal, setShowProductModal] = useState(false)
  const [showBrandModal, setShowBrandModal] = useState(false)
  const [showFacilityModal, setShowFacilityModal] = useState(false)
  const [search, setSearch] = useState("");
  const [category, setCategory] = useState("");
  const [sort, setSort] = useState("name-asc");

  useEffect(() => {
    const fetchData = async () => {
      setLoading(true);
      try {
        const [productsData, brandsData, categoriesData, facilitiesData, homePageLayoutData] = await Promise.all([
          loadProducts(),
          loadBrands(),
          loadCategories(),
          loadFacilities(),
          getHomePageLayout()
        ]);
        setProducts(productsData);
        setBrands(brandsData);
        setCategories(categoriesData);
        setFacilities(facilitiesData);
        setHomePageLayout(homePageLayoutData);
      } catch (err) {
        console.error("Veri yüklenirken hata:", err);
      } finally {
        setLoading(false);
      }
    };
    fetchData();
  }, []);

  const handleRefresh = async () => {
    setLoading(true);
    try {
      const [productsData, brandsData, categoriesData, facilitiesData, homePageLayoutData] = await Promise.all([
        loadProducts(),
        loadBrands(),
        loadCategories(),
        loadFacilities(),
        getHomePageLayout()
      ]);
      setProducts(productsData);
      setBrands(brandsData);
      setCategories(categoriesData);
      setFacilities(facilitiesData);
      setHomePageLayout(homePageLayoutData);
    } catch (err) {
      console.error("Veri yenilenirken hata:", err);
    } finally {
      setLoading(false);
    }
  };

  const handleAddProduct = () => {
    setSelectedProduct(null)
    setShowProductModal(true)
  }

  const handleEditProduct = (productToEdit: Product) => {
    const productFound = products.find(p => p.id === productToEdit.id);
    setSelectedProduct(productFound || null);
    setShowProductModal(true);
  }

  const handleDeleteProduct = async (id: string) => {
    setProducts((prev) => prev.filter((p) => p.id !== id))
    // Firebase'den silme işlemi ProductsTab'da yapılacak
  }

  const handleSaveProduct = (product: Product) => {
    if (selectedProduct) {
      setProducts((prev) => prev.map((p) => (p.id === product.id ? product : p)))
    } else {
      setProducts((prev) => [...prev, product])
    }
    setShowProductModal(false)
  }

  const handleAddBrand = () => {
    setSelectedBrand(null)
    setShowBrandModal(true)
  }

  const handleEditBrand = (brand: Brand) => {
    setSelectedBrand(brand)
    setShowBrandModal(true)
  }

  const handleSaveBrand = (brand: Brand) => {
    if (selectedBrand) {
      setBrands((prev) => prev.map((b) => (b.id === brand.id ? brand : b)))
    } else {
      setBrands((prev) => [...prev, brand])
    }
    handleRefresh() // Veriyi yeniden yükle
    setShowBrandModal(false)
  }

  const handleAddFacility = () => {
    setSelectedFacility(null)
    setShowFacilityModal(true)
  }

  const handleEditFacility = (facility: Facility) => {
    setSelectedFacility(facility)
    setShowFacilityModal(true)
  }

  const handleSaveFacility = (facility: Facility) => {
    if (selectedFacility) {
      setFacilities((prev) => prev.map((f) => (f.id === facility.id ? facility : f)))
    } else {
      setFacilities((prev) => [...prev, facility])
    }
    setShowFacilityModal(false)
    handleRefresh()
  }

  const handleDeleteFacility = (id: string) => {
    setFacilities((prev) => prev.filter((f) => f.id !== id))
  }

  const stats = {
    totalProducts: products.length,
    totalBrands: brands.length,
    mostLiked: products.reduce((max, p) => (p.likes > max.likes ? p : max), products[0] || { likes: 0 }),
    avgRating: products.reduce((sum, p) => sum + p.rating, 0) / products.length || 0,
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <header className="bg-white shadow-sm border-b">
        <div className="container mx-auto px-6 py-4">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-2xl font-bold text-blue-600">Ürün Yönetim Paneli</h1>
              <p className="text-gray-600">Vitamin & Takviye Ürünleri</p>
            </div>
            <div className="flex items-center gap-4">
              <Button variant="outline" asChild>
                <a href="/">
                  <Package className="h-4 w-4 mr-2" />
                  Ana Sayfa
                </a>
              </Button>
              <Button onClick={handleAddProduct}>
                <Plus className="h-4 w-4 mr-2" />
                Yeni Ürün Ekle
              </Button>
            </div>
          </div>
        </div>
      </header>

      {/* Main Content */}
      <main className="container mx-auto px-6 py-8">
        {/* Quick Stats */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Toplam Ürün</CardTitle>
              <Package className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-blue-600">{stats.totalProducts}</div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Toplam Marka</CardTitle>
              <Users className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-green-600">{stats.totalBrands}</div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Ortalama Puan</CardTitle>
              <TrendingUp className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-yellow-600">{stats.avgRating.toFixed(1)}</div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">En Beğenilen</CardTitle>
              <BarChart3 className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-sm font-medium truncate">{stats.mostLiked?.name || "-"}</div>
            </CardContent>
          </Card>
        </div>

        {/* Tabs */}
        <Tabs defaultValue="products" className="space-y-6">
          <TabsList className="grid w-full grid-cols-10">
            <TabsTrigger value="products">Ürünler</TabsTrigger>
            <TabsTrigger value="brands">Markalar</TabsTrigger>
            <TabsTrigger value="facilities">Üretim Tesisleri</TabsTrigger>
            <TabsTrigger value="categories">Kategori Yönetimi</TabsTrigger>
            <TabsTrigger value="requests">Ürün Talepleri</TabsTrigger>
            <TabsTrigger value="stats">İstatistikler</TabsTrigger>
            <TabsTrigger value="json">JSON</TabsTrigger>
            <TabsTrigger value="homepage">Anasayfa Düzeni</TabsTrigger>
            <TabsTrigger value="carouselTypes">Karosel Türleri</TabsTrigger>
            <TabsTrigger value="processor">İşleyici</TabsTrigger>
          </TabsList>

          <TabsContent value="products">
            <ProductsTab
              products={products}
              loading={loading}
              search={search}
              category={category}
              sort={sort}
              setSearch={setSearch}
              setCategory={setCategory}
              setSort={setSort}
              onEdit={handleEditProduct}
              onDelete={handleDeleteProduct}
              onAdd={handleAddProduct}
              onRefresh={handleRefresh}
              categories={categories}
            />
          </TabsContent>

          <TabsContent value="brands">
            <BrandsTab
              brands={brands}
              loading={loading}
              onEdit={handleEditBrand}
              onDelete={(id) => {
                setBrands((prev) => prev.filter((b) => b.id !== id))
              }}
              onAdd={handleAddBrand}
              onRefresh={handleRefresh}
            />
          </TabsContent>

          <TabsContent value="facilities">
            <FacilitiesTab
              facilities={facilities}
              products={products}
              brands={brands}
              loading={loading}
              onEdit={handleEditFacility}
              onDelete={handleDeleteFacility}
              onAdd={handleAddFacility}
              onRefresh={handleRefresh}
            />
          </TabsContent>

          <TabsContent value="categories">
            <CategoriesTab />
          </TabsContent>

          <TabsContent value="requests">
            <ProductRequestsTab categories={categories} />
          </TabsContent>

          <TabsContent value="stats">
            <StatsTab products={products} brands={brands} categories={categories} />
          </TabsContent>

          <TabsContent value="json">
            <JsonTab products={products} brands={brands} facilities={facilities} onRefresh={handleRefresh} />
          </TabsContent>

          

          

          

          <TabsContent value="homepage">
            <HomePageLayoutTab 
              products={products}
              onSaveLayout={saveHomePageLayout}
              onGetLayout={getHomePageLayout}
            />
          </TabsContent>

          <TabsContent value="carouselTypes">
            <CustomWidgetsTab />
          </TabsContent>

          <TabsContent value="processor">
            <ProcessorTab />
          </TabsContent>
        </Tabs>
      </main>

      {/* Modals */}
      {showProductModal && (
        <ProductModal
          product={selectedProduct}
          onClose={() => setShowProductModal(false)}
          onSave={handleSaveProduct}
          categories={categories}
          allBrands={memoizedBrands}
        />
      )}

      {showBrandModal && (
        <BrandModal
          brand={selectedBrand}
          onClose={() => setShowBrandModal(false)}
          onSave={handleSaveBrand}
        />
      )}

      {showFacilityModal && (
        <FacilityModal
          facility={selectedFacility}
          onClose={() => setShowFacilityModal(false)}
          onSave={handleSaveFacility}
          brands={brands}
        />
      )}

      {isMobile && (
        <MobileBottomBar
          onSearch={() => {}}
          onCategoryChange={() => {}}
          filters={{
            category: "*",
            subCategories: [],
            brands: [],
            contents: [],
            rating: { green: false, red: false },
            search: "",
            yerliMarka: false,
          }}
          onFiltersChange={() => {}}
          categories={categories}
        />
      )}
    </div>
  )
}
