'use client'

import { useState, useEffect } from 'react'
import Header from '@/components/Header'
import MobileBottomBar from '@/components/MobileBottomBar'
import ProductGrid from '@/components/ProductGrid'
import Footer from '@/components/Footer'
import FeaturedProduct from '@/components/FeaturedProduct'
import ProductCarousel from '@/components/ProductCarousel'
import DoctorApprovedProductsCarousel from '@/components/DoctorApprovedProductsCarousel'
import TopQualityProduct from '@/components/TopQualityProduct'
import InfoCard from '@/components/InfoCard'
import type { Product, Category, Brand, Facility, HomePageSection, CustomWidgetTemplate } from '@/types'
import { loadProducts, loadCategories, loadBrands, loadFacilities, getHomePageLayout, loadCustomWidgetTemplates } from '@/lib/firebase'
import { useSearchParams, useRouter } from 'next/navigation'

export default function HomePageClient() {
  const [products, setProducts] = useState<Product[]>([])
  const [filteredProducts, setFilteredProducts] = useState<Product[]>([])
  const [productsLimit, setProductsLimit] = useState(20) // Reduced from 40 for better mobile performance
  const [loading, setLoading] = useState(true)
  const [categories, setCategories] = useState<Category[]>([])
  const [brands, setBrands] = useState<any[]>([])
  const [facilities, setFacilities] = useState<any[]>([])
  const [homePageLayout, setHomePageLayout] = useState<HomePageSection[]>([]);
  const [customWidgetTemplates, setCustomWidgetTemplates] = useState<CustomWidgetTemplate[]>([]);
  const [filters, setFilters] = useState<{
    category: string
    subCategories: string[]
    brands: string[]
    contents: string[]
    rating: { green: boolean; red: boolean }
    search: string
    yerliMarka: boolean
  }>({
    category: "*",
    subCategories: [],
    brands: [],
    contents: [],
    rating: { green: false, red: false },
    search: "",
    yerliMarka: false,
  })
  const [error, setError] = useState<string | null>(null)

  const searchParams = useSearchParams();
  const router = useRouter();

  useEffect(() => {
    const fetchData = async () => {
      setLoading(true);
      try {
        const [productsData, categoriesData, brandsData, facilitiesData, homePageLayoutData, customCarouselsData] = await Promise.all([
          loadProducts(),
          loadCategories(),
          loadBrands(),
          loadFacilities(),
          getHomePageLayout(),
          loadCustomWidgetTemplates()
        ]);
        setProducts(productsData);
        setCategories(categoriesData);
        setBrands(brandsData);
        setFacilities(facilitiesData);
        setHomePageLayout(homePageLayoutData);
        setCustomWidgetTemplates(customCarouselsData);

        const brandFromUrl = searchParams.get('brand');
        if (brandFromUrl) {
          setFilters(prev => ({ ...prev, brands: [decodeURIComponent(brandFromUrl)] }));
        }

      } catch (err) {
        console.error("Veri yüklenirken hata:", err);
        setError("Veri yüklenirken bir hata oluştu.");
      } finally {
        setLoading(false);
      }
    };
    fetchData();
  }, [searchParams]);

  useEffect(() => {
    applyFilters()
  }, [products, filters])

  const refreshProducts = async () => {
    setLoading(true);
    try {
      const productsData = await loadProducts();
      setProducts(productsData);
    } catch (err) {
      console.error("Ürünler yenilenirken hata:", err);
      setError("Ürünler yenilenirken bir hata oluştu.");
    } finally {
      setLoading(false);
    }
  };

  const applyFilters = () => {
    let filtered = [...products]

    if (filters.category !== "*") {
      filtered = filtered.filter((p) => p.category === filters.category)
    }
    if (filters.subCategories.length > 0) {
      filtered = filtered.filter((p) => p.subCategory && filters.subCategories.includes(p.subCategory))
    }
    if (filters.brands.length > 0) {
      filtered = filtered.filter((p) => {
        const brandName = typeof p.brand === 'string' ? p.brand : p.brand.name
        return filters.brands.includes(brandName)
      })
    }
    if (filters.yerliMarka) {
      filtered = filtered.filter((p) => p.yerliMarka === true)
    }
    if (filters.search) {
      const searchLower = filters.search.toLowerCase()
      filtered = filtered.filter(
        (p) => p.name.toLowerCase().includes(searchLower) ||
               (typeof p.brand === 'string' ? p.brand.toLowerCase().includes(searchLower) : p.brand.name.toLowerCase().includes(searchLower)),
      )
    }
    setFilteredProducts(filtered)
  }

  const handleLoadMore = () => {
    setProductsLimit(prevLimit => prevLimit + 20) // Reduced increment for mobile
  }

  const handleCategoryChange = (category: string) => {
    if (searchParams.has('brand')) {
      router.push('/');
    }
    setFilters(prev => ({
      ...prev,
      category: category,
      brands: [],
      search: "",
    }));
  };

  // Get featured sections from layout
  const featuredSection = homePageLayout.find(s => s.type === "featured_product");
  const topQualitySection = homePageLayout.find(s => s.type === "custom_widget" &&
    customWidgetTemplates.find(t => t.id === s.sectionId)?.type === "single_product");
  const carouselSections = homePageLayout.filter(s => s.type === "product_carousel" ||
    (s.type === "custom_widget" && customWidgetTemplates.find(t => t.id === s.sectionId)?.type === "product_carousel"));

  const featuredProduct = featuredSection ? products.find(p => p.id === featuredSection.productId) : null;
  const topQualityProduct = topQualitySection ? (() => {
    const template = customWidgetTemplates.find(t => t.id === topQualitySection.sectionId);
    return template ? products.find(p => p.id === template.productId) : null;
  })() : null;

  return (
    <div className="min-h-screen bg-gray-50 flex flex-col">
      <Header
        onSearch={(query) => setFilters((prev) => ({ ...prev, search: query }))}
        onCategoryChange={handleCategoryChange}
        categories={categories}
        products={products}
        brands={brands}
        facilities={facilities}
      />

      <main className="flex-1 w-full">
        {error && (
          <div className="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative mx-4 mt-4" role="alert">
            {error}
          </div>
        )}

        {/* Hero Section - Only on Desktop */}
        <div className="hidden md:block">
          <div className="container mx-auto px-4 py-6">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
              {featuredProduct && (
                <div className="lg:col-span-1">
                  <FeaturedProduct product={featuredProduct} />
                </div>
              )}
              {topQualityProduct && (
                <div className="lg:col-span-1">
                  <TopQualityProduct product={topQualityProduct} />
                </div>
              )}
            </div>
          </div>
        </div>

        {/* Mobile Hero - Simplified */}
        <div className="md:hidden px-4 py-4">
          {topQualityProduct && (
            <div className="mb-4">
              <TopQualityProduct product={topQualityProduct} />
            </div>
          )}
        </div>

        {/* Carousels */}
        <div className="space-y-6 mb-8">
          {carouselSections.slice(0, 2).map((section) => {
            if (section.type === "product_carousel") {
              let carouselProducts = [...products];
              if (section.filter === "most_liked") {
                carouselProducts.sort((a, b) => b.likes - a.likes);
              } else if (section.filter === "newest") {
                carouselProducts.sort((a, b) => (new Date(b.createdAt || 0).getTime() - new Date(a.createdAt || 0).getTime()));
              } else if (section.filter === "custom_products" && section.productIds) {
                carouselProducts = products.filter(p => section.productIds!.includes(p.id));
              }
              return (
                <div key={section.id} className="px-4">
                  <ProductCarousel
                    title={section.title || "Öne Çıkanlar"}
                    products={carouselProducts.slice(0, 10)}
                    onProductClick={() => {}}
                    categories={categories}
                  />
                </div>
              );
            } else if (section.type === "custom_widget") {
              const template = customWidgetTemplates.find(t => t.id === section.sectionId);
              if (template?.type === "product_carousel") {
                const carouselProducts = products.filter(p => template.productIds?.includes(p.id));
                return (
                  <div key={section.id}>
                    <DoctorApprovedProductsCarousel
                      products={carouselProducts}
                      bgColor={template.bgColor}
                      onProductClick={() => {}}
                      widgetTags={template.widgetTags}
                      categories={categories}
                    />
                  </div>
                );
              }
            }
            return null;
          })}
        </div>

        {/* Main Products Section */}
        <div className="px-4 pb-24">
          <div className="mb-4">
            <h2 className="text-xl font-bold text-gray-900 mb-2">
              {filters.category !== "*" ? `${filters.category} Ürünleri` : "Tüm Ürünler"}
            </h2>
            <p className="text-sm text-gray-600">
              {filteredProducts.length} ürün bulundu
            </p>
          </div>

          <ProductGrid
            products={filteredProducts.slice(0, productsLimit)}
            loading={loading}
            onRefresh={refreshProducts}
            categories={categories}
            brands={brands}
            onLoadMore={handleLoadMore}
            hasMore={filteredProducts.length > productsLimit}
          />
        </div>
      </main>

      <Footer />
      <MobileBottomBar
        onSearch={(query) => setFilters((prev) => ({ ...prev, search: query }))}
        onCategoryChange={handleCategoryChange}
        filters={filters}
        onFiltersChange={setFilters}
        products={products}
        categories={categories}
      />
    </div>
  )
}
