"use client"

import { useState, useEffect } from "react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Accordion, AccordionItem, AccordionTrigger, AccordionContent } from "@/components/ui/accordion";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import ProductSelect from "@/components/admin/ProductSelect";
import { Product } from "@/types";
import { db, loadProducts } from "@/lib/firebase";
import { collection, addDoc, updateDoc, query, orderBy, onSnapshot, DocumentData, QueryDocumentSnapshot, doc, deleteDoc } from "firebase/firestore";

interface ProcessedProduct {
  id?: string; // Firestore document ID
  productId: string;
  productName: string;
  trendyolUrl: string;
  status: "Pending" | "Processing" | "Completed" | "Failed";
  timestamp: string;
  aiResponse?: any; // To store the AI's JSON response
  error?: string;
}

export default function ProcessorTab() {
  const [products, setProducts] = useState<Product[]>([]);
  const [selectedProductIds, setSelectedProductIds] = useState<string[]>([]);
  const [trendyolUrl, setTrendyolUrl] = useState<string>("");
  const [processedProducts, setProcessedProducts] = useState<ProcessedProduct[]>([]);
  const [isProcessing, setIsProcessing] = useState<boolean>(false);
  const [autoProcessProductName, setAutoProcessProductName] = useState<string>("");
  

  useEffect(() => {
    const fetchProducts = async () => {
      const productsData = await loadProducts();
      setProducts(productsData);
    };
    fetchProducts();

    // Firestore'dan işlenmiş ürünleri dinle
    const q = query(collection(db, "processedTasks"), orderBy("timestamp", "asc")); // Order by ascending for queue
    const unsubscribe = onSnapshot(q, (snapshot) => {
      const tasks: ProcessedProduct[] = [];
      snapshot.forEach((doc: QueryDocumentSnapshot<DocumentData>) => {
        tasks.push({ id: doc.id, ...doc.data() } as ProcessedProduct);
      });
      setProcessedProducts(tasks);
    });

    // Görev işleyici
    const processNextTask = async () => {
      // Check if there's any task currently being processed
      const processingTask = processedProducts.find(t => t.status === "Processing");
      if (processingTask) {
        return; // A task is already being processed, wait for it to complete
      }

      // Find the next pending task
      const pendingTask = processedProducts.find(t => t.status === "Pending");
      if (!pendingTask) {
        return; // No pending tasks
      }

      // Update task status to "Processing"
      const taskRef = doc(db, "processedTasks", pendingTask.id as string);
      await updateDoc(taskRef, { status: "Processing" });

      try {
        const response = await fetch("/api/process-product", {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
          },
          body: JSON.stringify({ productId: pendingTask.productId, trendyolUrl: pendingTask.trendyolUrl }),
        });

        const result = await response.json();

        await updateDoc(taskRef, {
          status: response.ok ? "Completed" : "Failed",
          aiResponse: response.ok ? result : null,
          error: response.ok ? null : (result.error || "Bilinmeyen hata"),
        });

      } catch (error: any) {
        console.error("Ürün işlenirken hata:", error);
        await updateDoc(taskRef, {
          status: "Failed",
          error: (error instanceof Error) ? error.message : String(error || "Bilinmeyen ağ hatası"),
        });
      }
    };

    // Call processNextTask whenever processedProducts changes
    processNextTask();

    // Cleanup function
    return () => unsubscribe();
  }, [processedProducts]); // Depend on processedProducts to trigger processing

  const handleProcessProduct = async () => {
    if (selectedProductIds.length === 0 || !trendyolUrl) {
      alert("Lütfen bir ürün seçin ve Trendyol URL'sini girin.");
      return;
    }

    const productToProcess = products.find(p => p.id === selectedProductIds[0]);

    if (!productToProcess) {
      alert("Seçilen ürün bulunamadı.");
      return;
    }

    const newProcessedProduct: Omit<ProcessedProduct, 'id'> = { // Omit 'id' as it's generated by Firestore
      productId: productToProcess.id,
      productName: productToProcess.name,
      trendyolUrl: trendyolUrl,
      status: "Pending",
      timestamp: new Date().toISOString(), // Use ISO string for consistent timestamp
    };

    try {
      // Add new task to Firestore with "Pending" status
      await addDoc(collection(db, "processedTasks"), newProcessedProduct);
      console.log("Yeni görev sıraya eklendi:", newProcessedProduct.productName);
    } catch (error: any) {
      console.error("Görevi sıraya eklerken hata:", error);
      alert("Görevi sıraya eklerken hata oluştu: " + ((error instanceof Error) ? error.message : String(error)));
    } finally {
      setSelectedProductIds([]);
      setTrendyolUrl("");
    }
  };

  const handleAutoProcessProduct = async (product: Product) => {
    if (!product.trendyolUrl) {
      alert(`'${product.name}' ürünü için Trendyol URL'si bulunamadı.`);
      return;
    }

    const newProcessedProduct: Omit<ProcessedProduct, 'id'> = {
      productId: product.id,
      productName: product.name,
      trendyolUrl: product.trendyolUrl,
      status: "Pending",
      timestamp: new Date().toISOString(),
    };

    try {
      await addDoc(collection(db, "processedTasks"), newProcessedProduct);
      console.log("Yeni otomatik görev sıraya eklendi:", newProcessedProduct.productName);
    } catch (error: any) {
      console.error("Otomatik görevi sıraya eklerken hata:", error);
      alert("Otomatik görevi sıraya eklerken hata oluştu: " + ((error instanceof Error) ? error.message : String(error)));
    }
  };

  const handleProcessAllProducts = async () => {
    if (products.length === 0) {
      alert("İşlenecek ürün bulunamadı.");
      return;
    }
    for (const product of products) {
      if (product.trendyolUrl) {
        await handleAutoProcessProduct(product);
      } else {
        console.warn(`'${product.name}' ürünü için Trendyol URL'si bulunamadığından otomatik işleme atlandı.`);
      }
    }
    alert("Tüm uygun ürünler otomatik işleme sırasına alındı.");
  };

  const handleCleanProducts = async () => {
    if (processedProducts.length === 0) {
      alert("Temizlenecek ürün bulunamadı.");
      return;
    }

    const productNamesSeen = new Set<string>();
    const productsToDelete: ProcessedProduct[] = [];

    processedProducts.forEach(product => {
      if (productNamesSeen.has(product.productName)) {
        productsToDelete.push(product);
      } else {
        productNamesSeen.add(product.productName);
      }
    });

    if (productsToDelete.length === 0) {
      alert("Mükerrer ürün bulunamadı.");
      return;
    }

    if (!confirm(`Toplam ${productsToDelete.length} adet mükerrer ürün silinecek. Emin misiniz?`)) {
      return;
    }

    let deletedCount = 0;
    for (const product of productsToDelete) {
      try {
        if (product.id) {
          await deleteDoc(doc(db, "processedTasks", product.id));
          deletedCount++;
        }
      } catch (error) {
        console.error(`Ürün silinirken hata oluştu: ${product.productName} (ID: ${product.id})`, error);
      }
    }
    alert(`${deletedCount} adet mükerrer ürün başarıyla silindi.`);
  };

  return (
    <Tabs defaultValue="product-processor" className="space-y-6">
      <TabsList>
        <TabsTrigger value="product-processor">Ürün İşleyici</TabsTrigger>
        <TabsTrigger value="auto-process">Auto Process</TabsTrigger>
      </TabsList>
      <TabsContent value="product-processor">
        <Card>
          <CardHeader>
            <CardTitle>Ürün İşleyici</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div>
              <Label htmlFor="product-select">Ürün Seç</Label>
              <ProductSelect
                allProducts={products}
                selectedProductIds={selectedProductIds}
                onProductIdsChange={setSelectedProductIds}
              />
            </div>
            <div>
              <Label htmlFor="trendyol-url">Trendyol Ürün Linki</Label>
              <Input
                id="trendyol-url"
                placeholder="Trendyol ürün linkini buraya yapıştırın"
                value={trendyolUrl}
                onChange={(e) => setTrendyolUrl(e.target.value)}
              />
            </div>
            <Button onClick={handleProcessProduct} disabled={isProcessing}>
              {isProcessing ? "İşleniyor..." : "İşlem Ekle"}
            </Button>
          </CardContent>
        </Card>

        <Tabs defaultValue="current-items" className="space-y-4"> 
          <TabsList>
            <TabsTrigger value="current-items">Sıradaki İşlemler</TabsTrigger> 
            <TabsTrigger value="processed-items">Yapılan İşlemler</TabsTrigger>
          </TabsList>
          <TabsContent value="current-items"> 
            <Card>
              <CardHeader>
                <CardTitle>Sıradaki İşlemler</CardTitle>
              </CardHeader>
              <CardContent>
                {processedProducts.filter(p => p.status === "Pending" || p.status === "Processing").length === 0 ? (
                  <p className="text-gray-500">Henüz sıradaki ürün yok.</p>
                ) : (
                  <Accordion type="single" collapsible className="w-full">
                    {processedProducts.filter(p => p.status === "Pending" || p.status === "Processing").map((item, index) => (
                      <AccordionItem key={item.id || index} value={`item-${item.id || index}`}>
                        <AccordionTrigger>
                          <div className="flex justify-between items-center w-full pr-4">
                            <span className="font-semibold">{item.productName}</span>
                            <span className={`font-semibold ${
                              item.status === "Completed" ? "text-green-600" :
                              item.status === "Failed" ? "text-red-600" :
                              "text-yellow-600"
                            }`}>{item.status}</span>
                          </div>
                        </AccordionTrigger>
                        <AccordionContent>
                          <div className="border p-4 rounded-md shadow-sm space-y-2">
                            <p><strong>Trendyol URL:</strong> <a href={item.trendyolUrl} target="_blank" rel="noopener noreferrer" className="text-blue-500 hover:underline">{item.trendyolUrl}</a></p>
                            <p><strong>Zaman:</strong> {item.timestamp}</p>
                            {item.error && <p className="text-red-500"><strong>Hata:</strong> {item.error}</p>}
                            {item.aiResponse && (
                              <div className="mt-2">
                                <h4 className="font-semibold">AI Yanıtı:</h4>
                                <pre className="bg-gray-100 p-2 rounded-md text-sm overflow-auto">
                                  {JSON.stringify(item.aiResponse, null, 2)}
                                </pre>
                              </div>
                            )}
                          </div>
                        </AccordionContent>
                      </AccordionItem>
                    ))}
                  </Accordion>
                )}
              </CardContent>
            </Card>
          </TabsContent>
          <TabsContent value="processed-items">
            <Card>
              <CardHeader>
                <CardTitle>Yapılan İşlemler</CardTitle>
              </CardHeader>
              <CardContent>
                {processedProducts.filter(p => p.status === "Completed" || p.status === "Failed").length === 0 ? (
                  <p className="text-gray-500">Henüz işlenmiş ürün yok.</p>
                ) : (
                  <Accordion type="single" collapsible className="w-full">
                    {processedProducts.filter(p => p.status === "Completed" || p.status === "Failed").map((item, index) => (
                      <AccordionItem key={item.id || index} value={`item-${item.id || index}`}>
                        <AccordionTrigger>
                          <div className="flex justify-between items-center w-full pr-4">
                            <span className="font-semibold">{item.productName}</span>
                            <span className={`font-semibold ${
                              item.status === "Completed" ? "text-green-600" :
                              item.status === "Failed" ? "text-red-600" :
                              "text-yellow-600"
                            }`}>{item.status}</span>
                          </div>
                        </AccordionTrigger>
                        <AccordionContent>
                          <div className="border p-4 rounded-md shadow-sm space-y-2">
                            <p><strong>Trendyol URL:</strong> <a href={item.trendyolUrl} target="_blank" rel="noopener noreferrer" className="text-blue-500 hover:underline">{item.trendyolUrl}</a></p>
                            <p><strong>Zaman:</strong> {item.timestamp}</p>
                            {item.error && <p className="text-red-500"><strong>Hata:</strong> {item.error}</p>}
                            {item.aiResponse && (
                              <div className="mt-2">
                                <h4 className="font-semibold">AI Yanıtı:</h4>
                                <pre className="bg-gray-100 p-2 rounded-md text-sm overflow-auto">
                                  {JSON.stringify(item.aiResponse, null, 2)}
                                </pre>
                              </div>
                            )}
                          </div>
                        </AccordionContent>
                      </AccordionItem>
                    ))}
                  </Accordion>
                )}
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </TabsContent>
      <TabsContent value="auto-process">
        <Card>
          <CardHeader>
            <CardTitle>Otomatik Ürün İşleme</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <Button onClick={handleProcessAllProducts} className="w-full mb-4">
              Tümünü İşlem Sırasına Al
            </Button>
            <Button onClick={handleCleanProducts} className="w-full mb-4" variant="destructive">
              Ürünleri Temizle
            </Button>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              {products.map((product) => {
                const isProductProcessing = processedProducts.some(t => t.productId === product.id && (t.status === "Pending" || t.status === "Processing"));
                const hasTrendyolUrl = !!product.trendyolUrl;

                return (
                  <Card key={product.id} className="flex flex-col justify-between">
                    <CardHeader>
                      <CardTitle className="text-base">{product.name}</CardTitle>
                    </CardHeader>
                    <CardContent>
                      {hasTrendyolUrl ? (
                        <Button 
                          onClick={() => handleAutoProcessProduct(product)}
                          disabled={isProductProcessing}
                          className="w-full"
                        >
                          {isProductProcessing ? "İşleniyor..." : "Yeniden İşle"}
                        </Button>
                      ) : (
                        <Button 
                          disabled
                          className="w-full"
                          variant="outline"
                        >
                          Trendyol URL Yok
                        </Button>
                      )}
                    </CardContent>
                  </Card>
                );
              })}
            </div>
          </CardContent>
        </Card>
      </TabsContent>
    </Tabs>
  );
}
