import React, { useState } from 'react';
import { saveFacility, saveProduct, loadCategories, addCategory, addSubCategory, loadProducts } from '@/lib/firebase';
import { doc, collection } from 'firebase/firestore';
import { db } from '@/lib/firebase';

interface UretimTesisi {
  kayit_no: string;
  uretim_tesisi: string;
}

interface Urun {
  [key: string]: any;
  "Üretici İşletme Kayıt/Onay Numarası": string;
  "Durumu (İthal/Yerli)"?: string;
  "Temiz içerik"?: string;
  "AI skoru"?: string;
}

interface EslestirilmisTesis extends UretimTesisi {
  uretilenler: Urun[];
}

const extractKayitKodu = (kayit: string) => {
  if (!kayit || typeof kayit !== 'string') return '';
  return kayit.split(' ')[0]?.trim();
};

const extractMarka = (urunAdi: string | undefined) => {
  if (!urunAdi) return "Bilinmiyor";
  const marka = urunAdi.split(" ")[0]?.trim();
  return marka || "Bilinmiyor";
};

const mapFormType = (hal: string) => {
  if (!hal) return {};
  const form: any = {
    tablet: false, kapsul: false, toz: false, sase: false, gummy: false, sivi: false, granül: false, pastil: false
  };
  const lower = hal.toLowerCase();
  if (lower.includes('çiğnenebilir')) form.gummy = true;
  if (lower.includes('granül')) form.granül = true;
  if (lower.includes('pastil')) form.pastil = true;
  if (lower.includes('tablet')) form.tablet = true;
  if (lower.includes('kapsül')) form.kapsul = true;
  if (lower.includes('toz')) form.toz = true;
  if (lower.includes('sase')) form.sase = true;
  if (lower.includes('sıvı')) form.sivi = true;
  return form;
};

const mapAdditives = (bilesenler: any[]) => {
  const additives = { renklendirici: false, aroma: false, tatlandirici: false };
  if (!Array.isArray(bilesenler)) return additives;
  bilesenler.forEach(b => {
    const l = (b || '').toLowerCase();
    if (l.includes('aroma')) additives.aroma = true;
    if (l.includes('tatlandırıcı')) additives.tatlandirici = true;
    if (l.includes('renklendirici')) additives.renklendirici = true;
  });
  return additives;
};

const normalizeString = (str: string | undefined) => {
  if (!str || typeof str !== 'string') {
    console.error("normalizeString: Invalid input str:", str, "type:", typeof str); // Added error log
    return '';
  }
  // console.log("normalizeString: Processing str:", str, "type:", typeof str); // Optional: for debugging valid inputs
  return str.toLowerCase()
    .replace(/\s+/g, ' ') // Replace multiple spaces with single space
    .replace(/\s*(-ithalat|-uretim|-satis|-satis-ithalat|-satis-ihracat|-ithalat-ihracat|-merkez|-sube|-subesi|-merkez-ofis|-takviye|-takviye-edici|-takviye-edici-gida|-gida|-ilac|-kozmetik|-medikal|-sanayi|-ticaret|-limited|-anonim|-sirketi|-ltd|-sti|-a-s|-as|-ltd-sti|-a-s-ithalat|-a-s-uretim|-a-s-satis|-a-s-ihracat|-a-s-pazarlama|-a-s-dis-ticaret|-a-s-gida|-a-s-ilac|-a-s-kozmetik|-a-s-medikal|-a-s-sanayi|-a-s-ticaret|-a-s-limited|-a-s-anonim|-a-s-sirketi|-a-s-ltd|-a-s-sti|-a-s-a-s|-a-s-as|-a-s-ltd-sti|-a-s-a-s-ithalat|-a-s-a-s-uretim|-a-s-a-s-satis|-a-s-a-s-ihracat|-a-s-a-s-pazarlama|-a-s-a-s-dis-ticaret|-a-s-a-s-gida|-a-s-a-s-ilac|-a-s-a-s-kozmetik|-a-s-a-s-medikal|-a-s-a-s-sanayi|-a-s-a-s-ticaret|-a-s-a-s-limited|-a-s-a-s-anonim|-a-s-a-s-sirketi|-a-s-a-s-ltd|-a-s-a-s-sti|-a-s-a-s-a-s|-a-s-a-s-as|-a-s-a-s-ltd-sti)\b/g, '') // Remove common suffixes
    .replace(/[^a-z0-9ğüşöçıİĞÜŞÖÇ]/g, '') // Remove non-alphanumeric characters (keep Turkish chars)
    .trim();
};

const normalizeKayitNo = (kod: string) => {
  if (!kod || typeof kod !== 'string') return '';
  return kod.replace(/[^a-zA-Z0-9]/g, '').toLowerCase();
};

const JsonImporterTab: React.FC = () => {
  const [tesisler, setTesisler] = useState<UretimTesisi[]>([]);
  const [urunler, setUrunler] = useState<Urun[]>([]);
  const [eslestirilmis, setEslestirilmis] = useState<EslestirilmisTesis[]>([]);
  const [eksikBilgi, setEksikBilgi] = useState<Urun[]>([]);
  const [eslesmeyenUrunler, setEslesmeyenUrunler] = useState<Urun[]>([]);
  const [acikTesis, setAcikTesis] = useState<number | null>(null);
  const [kayitSonucu, setKayitSonucu] = useState<string>("");
  const [totalTesisCount, setTotalTesisCount] = useState<number>(0);
  const [processedTesisCount, setProcessedTesisCount] = useState<number>(0);
  const [totalUrunCount, setTotalUrunCount] = useState<number>(0);
  const [processedUrunCount, setProcessedUrunCount] = useState<number>(0);
  const [currentProcessingItem, setCurrentProcessingItem] = useState<string>("");
  const [importLog, setImportLog] = useState<string[]>([]);

  const handleTesislerUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;
    const reader = new FileReader();
    reader.onload = (evt) => {
      try {
        const json = JSON.parse(evt.target?.result as string);
        setTesisler(json);
      } catch {
        alert('Geçersiz JSON dosyası (Üretim Tesisleri)');
      }
    };
    reader.readAsText(file);
  };

  const handleUrunlerUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;
    const reader = new FileReader();
    reader.onload = (evt) => {
      try {
        const json = JSON.parse(evt.target?.result as string);
        // Her ürüne Marka ekle
        const urunlerWithMarka = json.map((u: any) => ({
          ...u,
          Marka: extractMarka(u["Ürün Adı"]),
        }));
        setUrunler(urunlerWithMarka);
      } catch {
        alert('Geçersiz JSON dosyası (Ürünler)');
      }
    };
    reader.readAsText(file);
  };

  const eslestir = () => {
    if (!tesisler.length || !urunler.length) return;
    // Üretici kodlarını normalize eden map
    const tesisMap = new Map<string, UretimTesisi>();
    tesisler.forEach((tesis) => {
      const normalizedTesisAdi = normalizeString(tesis.uretim_tesisi);
      if (normalizedTesisAdi) {
        tesisMap.set(normalizedTesisAdi, tesis);
      }
    });

    const eslesmisUrunler: { [normalizedTesisAdi: string]: Urun[] } = {};
    const eslesmeyen: Urun[] = [];

    urunler.forEach((u) => {
      const ureticiAdi = u["Üretici/İthalatçı İşletme(ler)"] || '';
      const normalizedUreticiAdi = normalizeString(ureticiAdi);

      if (normalizedUreticiAdi && tesisMap.has(normalizedUreticiAdi)) {
        if (!eslesmisUrunler[normalizedUreticiAdi]) eslesmisUrunler[normalizedUreticiAdi] = [];
        eslesmisUrunler[normalizedUreticiAdi].push({ ...u, matchedProducer: tesisMap.get(normalizedUreticiAdi) });
      } else {
        eslesmeyen.push(u);
      }
    });

    const sonuc: EslestirilmisTesis[] = Array.from(tesisMap.entries()).map(([normalizedTesisAdi, tesis]) => ({
      ...tesis,
      uretilenler: eslesmisUrunler[normalizedTesisAdi] || [],
    }));
    setEslestirilmis(sonuc);
    setEslesmeyenUrunler(eslesmeyen);
    eksikleriBul(urunler);
  };

  const eksikleriBul = (urunlerList: Urun[]) => {
    const eksikler = urunlerList.filter(
      (u) => !u["Temiz içerik"] || !u["AI skoru"]
    );
    setEksikBilgi(eksikler);
  };

  const handleGuncelle = () => {
    eksikleriBul(urunler);
  };

  // Tesis ve ürünleri db'ye kaydet/güncelle
  const kaydetVeGuncelle = async () => {
    if (!eslestirilmis.length) return;
    setTotalTesisCount(eslestirilmis.length);
    setTotalUrunCount(eslestirilmis.reduce((acc, tesis) => acc + tesis.uretilenler.length, 0));
    setProcessedTesisCount(0);
    setProcessedUrunCount(0);
    setCurrentProcessingItem("");
    setImportLog([]);

    let tesisOk = 0, urunOk = 0, tesisErr = 0, urunErr = 0;
    const categories = await loadCategories();
    const existingProducts = await loadProducts();
    const productMap = new Map<string, string>();
    existingProducts.forEach(p => {
      const key = `${normalizeString(p.name)}_${normalizeString(p.brand as string)}`;
      productMap.set(key, p.id);
    });

    for (const tesis of eslestirilmis) {
      setCurrentProcessingItem(`Tesis: ${tesis.uretim_tesisi}`);
      if (tesis.uretim_tesisi === "İDİ SOLUTİON Vitamin C, Sodyum ve Potasyum İçeren Şurup Takviye Edici Gıda") {
        console.warn("Tesis özel isimle eşleşti, atlanıyor:", tesis.uretim_tesisi);
        tesisErr++;
        setProcessedTesisCount(prev => prev + 1);
        continue;
      }
      if (!tesis.kayit_no || tesis.kayit_no.trim() === '') {
        console.warn("Tesisin kayıt numarası eksik veya geçersiz, atlanıyor:", tesis.uretim_tesisi);
        tesisErr++;
        setProcessedTesisCount(prev => prev + 1);
        continue;
      }
      if (typeof tesis.uretim_tesisi !== 'string' || tesis.uretim_tesisi.trim() === '') {
        console.warn("Tesis adı boş veya geçersiz, atlanıyor:", tesis.uretim_tesisi);
        tesisErr++;
        setProcessedTesisCount(prev => prev + 1);
        continue;
      }
      const puanlar = tesis.uretilenler
        .map(u => Number(u["AI skoru"]) || 0)
        .filter(n => !isNaN(n));
      const score = puanlar.length ? Math.round(puanlar.reduce((a, b) => a + b, 0) / puanlar.length) : 0;
      const facilityObj = {
        id: tesis.kayit_no,
        name: tesis.uretim_tesisi || "",
        score: score || 0,
        brandsProduced: Array.from(new Set(tesis.uretilenler.map(u => String(u["Marka"] || '')).filter(Boolean))),
      };
      try {
        await saveFacility(facilityObj);
        tesisOk++;
        setImportLog(prev => [...prev, `Tesis kaydedildi: ${tesis.uretim_tesisi}`]);
      } catch (e: any) {
        console.error("Tesis kaydedilirken hata:", tesis.uretim_tesisi, e);
        tesisErr++;
      }
      setProcessedTesisCount(prev => prev + 1);

      for (const urun of tesis.uretilenler) {
        setCurrentProcessingItem(`Ürün: ${urun["Ürün Adı"]}`);
        const form = mapFormType(urun["Hal"]);
        const icerik = (urun["Bilesenler"] || []).map((b: any) => String(b || ''));
        const aiScore = Number(urun["Yapay_Zeka_Skoru"] || urun["AI skoru"] || 0);
        const urls = (urun["Ürünün Satışının Yapıldığı Beyan Edilen İnternet/URL Adresleri"] || '').split(',').map((u: string) => u.trim()).filter(Boolean);
        const buyLink = urls[0] || '';
        const durum = urun["Durumu (İthal/Yerli)"];
        const yerliMarka = typeof durum === "string" && durum.includes("Yerli");
        const uretimTesisiKayitNo = urun.matchedProducer?.kayit_no || '';

        let categoryId = '';
        let subCategoryId = '';
        const kategoriAdi = urun["Kategori"]?.trim();
        const altKategoriAdi = urun["Alt_Kategori"]?.trim();
        const urunAdi = urun["Ürün Adı"]?.trim();

        if (typeof urunAdi !== 'string' || urunAdi.trim() === '') {
          console.warn("Ürün adı boş veya geçersiz, atlanıyor:", urun["Ürün Adı"]);
          urunErr++;
          setProcessedUrunCount(prev => prev + 1);
          continue;
        }

        if (urunAdi === "İDİ SOLUTİON Vitamin C, Sodyum ve Potasyum İçeren Şurup Takviye Edici Gıda") {
          console.warn("Ürün özel isimle eşleşti, atlanıyor:", urunAdi);
          urunErr++;
          setProcessedUrunCount(prev => prev + 1);
          continue;
        }

        const productKey = `${normalizeString(urunAdi)}_${normalizeString(urun["Marka"])}`;
        const productId = productMap.get(productKey) || doc(collection(db, 'products')).id;

        const productObj: any = {
          id: productId,
          name: String(urunAdi || ""),
          brand: String(urun["Marka"] || ""),
          category: String(categoryId),
          subCategory: String(subCategoryId || ""),
          price: 0,
          rating: 0,
          likes: 0,
          dislikes: 0,
          image: "",
          evaluation: {
            ai_guven_skoru: aiScore,
          },
          cleanFactors: {
            form,
            additives: mapAdditives(urun["Bilesenler"])
          },
          icerik: (urun["Bilesenler"] || []).map((b: any) => String(b || '')),
          usage: String(urun["Kullanım Talimatı"] || ""),
          buyLink: String(buyLink),
          yerliMarka,
          uretimTesisiKayitNo: String(uretimTesisiKayitNo),
          onayNo: String(urun["Üretici İşletme Kayıt/Onay Numarası"] || ''),
        };

        try {
          await saveProduct(productObj);
          urunOk++;
          setImportLog(prev => [...prev, `Ürün kaydedildi: ${urunAdi}`]);
        } catch (e: any) {
          console.error("Ürün kaydedilirken hata:", urun["Ürün Adı"], e);
          urunErr++;
        }
        setProcessedUrunCount(prev => prev + 1);
      }
    }
    setKayitSonucu(`${urunOk} ürün, ${tesisOk} tesis kaydedildi. Hatalı ürün: ${urunErr}, hatalı tesis: ${tesisErr}`);
    setCurrentProcessingItem("İşlem tamamlandı.");
  };

  return (
    <div className="p-6 max-w-3xl mx-auto">
      <h2 className="text-2xl font-bold mb-4">JSON (Tarım Ve Orman Bakanlığı)</h2>
      <div className="flex flex-col gap-4 mb-6">
        <label className="block">
          <span className="font-medium">Üretim Tesisleri JSON Yükle</span>
          <input type="file" accept="application/json" onChange={handleTesislerUpload} className="mt-1" />
        </label>
        <label className="block">
          <span className="font-medium">Ürünler JSON Yükle</span>
          <input type="file" accept="application/json" onChange={handleUrunlerUpload} className="mt-1" />
        </label>
        <button
          className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700 w-fit"
          onClick={eslestir}
          disabled={!tesisler.length || !urunler.length}
        >
          Eşleştir ve Göster
        </button>
      </div>

      {eslestirilmis.length > 0 && (
        <div className="mb-8">
          <h3 className="text-xl font-semibold mb-2">Eşleştirilmiş Tesisler ve Ürünler</h3>
          <button
            className="bg-green-600 text-white px-4 py-2 rounded hover:bg-green-700 mb-4"
            onClick={kaydetVeGuncelle}
          >
            Tesis ve Ürünleri Kaydet / Güncelle
          </button>
          {kayitSonucu && <div className="text-sm text-green-700 mb-2">{kayitSonucu}</div>}

          {/* Import Progress Section */}
          {(totalTesisCount > 0 || totalUrunCount > 0) && (
            <div className="mt-4 p-4 border rounded bg-gray-50">
              <h4 className="text-lg font-semibold mb-2">İmport İlerlemesi</h4>
              <p className="text-sm">Şu an işleniyor: {currentProcessingItem}</p>
              <div className="mt-2">
                <p className="text-sm">Tesisler: {processedTesisCount} / {totalTesisCount} ({(totalTesisCount > 0 ? (processedTesisCount / totalTesisCount * 100) : 0).toFixed(2)}%)</p>
                <div className="w-full bg-gray-200 rounded-full h-2.5 dark:bg-gray-700">
                  <div className="bg-blue-600 h-2.5 rounded-full" style={{ width: `${(totalTesisCount > 0 ? (processedTesisCount / totalTesisCount * 100) : 0)}%` }}></div>
                </div>
              </div>
              <div className="mt-2">
                <p className="text-sm">Ürünler: {processedUrunCount} / {totalUrunCount} ({(totalUrunCount > 0 ? (processedUrunCount / totalUrunCount * 100) : 0).toFixed(2)}%)</p>
                <div className="w-full bg-gray-200 rounded-full h-2.5 dark:bg-gray-700">
                  <div className="bg-blue-600 h-2.5 rounded-full" style={{ width: `${(totalUrunCount > 0 ? (processedUrunCount / totalUrunCount * 100) : 0)}%` }}></div>
                </div>
              </div>
              <div className="mt-4">
                <h5 className="text-md font-semibold mb-1">İmport Logu:</h5>
                <div className="max-h-40 overflow-y-auto border p-2 rounded bg-white text-xs">
                  {importLog.length === 0 ? (
                    <p>Log boş.</p>
                  ) : (
                    importLog.map((log, index) => (
                      <p key={index}>{log}</p>
                    ))
                  )}
                </div>
              </div>
            </div>
          )}

          <div className="space-y-6">
            {eslestirilmis.map((tesis, i) => (
              <div key={i} className="border rounded p-4 bg-gray-50">
                <div className="flex items-center justify-between">
                  <div>
                    <div className="font-bold text-lg mb-1">{tesis.uretim_tesisi}</div>
                    <div className="text-sm text-gray-600 mb-2">Kayıt No: {tesis.kayit_no}</div>
                  </div>
                  <button
                    className="bg-gray-200 px-3 py-1 rounded text-sm hover:bg-gray-300"
                    onClick={() => setAcikTesis(acikTesis === i ? null : i)}
                  >
                    {tesis.uretilenler.length} ürün
                  </button>
                </div>
                {acikTesis === i && (
                  <div className="mt-4 grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-3 max-h-96 overflow-y-auto">
                    {tesis.uretilenler.length === 0 && (
                      <div className="text-gray-400">Ürün bulunamadı.</div>
                    )}
                    {tesis.uretilenler.map((urun, j) => (
                      <div
                        key={j}
                        className="bg-white border rounded shadow-sm p-2 flex flex-col gap-1 text-xs hover:shadow-md transition min-h-[80px]"
                      >
                        <div className="font-semibold text-sm truncate" title={urun["Ürün Adı"]}>{urun["Ürün Adı"]}</div>
                        {urun["Marka"] && (
                          <div className="text-gray-700">{urun["Marka"]}</div>
                        )}
                        {urun["Durumu (İthal/Yerli)"] && (
                          <div className="text-gray-500">{urun["Durumu (İthal/Yerli)"]}</div>
                        )}
                        {urun["Üretici İşletme Kayıt/Onay Numarası"] && (
                          <div className="text-gray-400">Kayıt No: {urun["Üretici İşletme Kayıt/Onay Numarası"]}</div>
                        )}
                        {/* Eksik bilgi uyarısı */}
                        {(!urun["Temiz içerik"] || !urun["AI skoru"]) && (
                          <div className="text-red-500 mt-1">Eksik: {(!urun["Temiz içerik"] ? 'Temiz içerik ' : '') + (!urun["AI skoru"] ? 'AI skoru' : '')}</div>
                        )}
                      </div>
                    ))}
                  </div>
                )}
              </div>
            ))}
          </div>
        </div>
      )}

      {/* Eşleşmeyen Ürünler */}
      {eslesmeyenUrunler.length > 0 && (
        <div className="mb-8">
          <h3 className="text-xl font-semibold text-red-600 mb-2">Eşleşmeyen Ürünler</h3>
          <ul className="list-disc ml-6">
            {eslesmeyenUrunler.map((urun, i) => (
              <li key={i} className="mb-1">
                <span className="font-semibold">{urun["Ürün Adı"]}</span>
                <span className="ml-2 text-xs text-gray-500">Kayıt No: {urun["Üretici İşletme Kayıt/Onay Numarası"] || <span className='text-red-400'>YOK</span>}</span>
              </li>
            ))}
          </ul>
        </div>
      )}

      <div className="mb-8">
        <div className="flex items-center gap-2 mb-2">
          <h3 className="text-xl font-semibold">Eksik Bilgi Olan Ürünler</h3>
          <button
            className="bg-gray-200 px-2 py-1 rounded text-sm hover:bg-gray-300"
            onClick={handleGuncelle}
          >
            Güncelle
          </button>
        </div>
        {eksikBilgi.length === 0 ? (
          <div className="text-green-600">Tüm ürünlerde Temiz içerik ve AI skoru mevcut.</div>
        ) : (
          <ul className="list-disc ml-6">
            {eksikBilgi.map((urun, i) => (
              <li key={i} className="mb-1">
                <span className="font-semibold">{urun["Ürün Adı"]}</span>
                <span className="ml-2 text-xs text-gray-500">Kayıt No: {urun["Üretici İşletme Kayıt/Onay Numarası"]}</span>
                <span className="ml-2 text-xs text-red-500">Eksik: {(!urun["Temiz içerik"] ? 'Temiz içerik ' : '') + (!urun["AI skoru"] ? 'AI skoru' : '')}</span>
              </li>
            ))}
          </ul>
        )}
      </div>
    </div>
  );
};

export default JsonImporterTab;