"use client"

import { useState } from "react"
import { Database, Download, FileText, Upload } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import type { Product, Brand, Facility } from "@/types"
import { saveProduct, saveBrand, saveFacility } from "@/lib/firebase"

interface JsonExporterProps {
  products: Product[]
  brands: Brand[]
  facilities: Facility[]
  onRefresh: () => void
}

export default function JsonExporter({ products, brands, facilities, onRefresh }: JsonExporterProps) {
  const [exporting, setExporting] = useState(false)
  const [importing, setImporting] = useState(false)
  const [selectedFile, setSelectedFile] = useState<File | null>(null)

  const downloadJson = (data: any, filename: string) => {
    const jsonString = JSON.stringify(data, null, 2)
    const blob = new Blob([jsonString], { type: "application/json" })
    const url = URL.createObjectURL(blob)
    const link = document.createElement("a")
    link.href = url
    link.download = filename
    document.body.appendChild(link)
    link.click()
    document.body.removeChild(link)
    URL.revokeObjectURL(url)
  }

  const exportProducts = () => {
    setExporting(true)
    try {
      const exportData = products.map(product => ({
        id: product.id,
        name: product.name,
        brand: product.brand,
        category: product.category,
        subCategory: product.subCategory,
        price: product.price,
        image: product.image,
        rating: product.rating,
        likes: product.likes,
        dislikes: product.dislikes,
        evaluation: {
          temiz_icerik: product.evaluation?.temiz_icerik || 0,
          formul_kalitesi: product.evaluation?.formul_kalitesi || 0,
          uretim_tesisi: product.evaluation?.uretim_tesisi || 0,
          kullanici_yorumlari: product.evaluation?.kullanici_yorumlari || 0,
          uretim_sertifika: product.evaluation?.uretim_sertifika || 0,
          ai_guven_skoru: product.evaluation?.ai_guven_skoru || 0,
          dradvice: product.evaluation?.dradvice || false,
          certFactors: {
            gmp: product.evaluation?.certFactors?.gmp || false,
            haccp: product.evaluation?.certFactors?.haccp || false,
            iso9001: product.evaluation?.certFactors?.iso9001 || false,
            iso22000: product.evaluation?.certFactors?.iso22000 || false,
          }
        },
        cleanFactors: {
          form: {
            tablet: product.cleanFactors?.form?.tablet || false,
            kapsul: product.cleanFactors?.form?.kapsul || false,
            toz: product.cleanFactors?.form?.toz || false,
            sase: product.cleanFactors?.form?.sase || false,
            gummy: product.cleanFactors?.form?.gummy || false,
            sivi: product.cleanFactors?.form?.sivi || false,
          },
          additives: {
            renklendirici: product.cleanFactors?.additives?.renklendirici || false,
            aroma: product.cleanFactors?.additives?.aroma || false,
            tatlandirici: product.cleanFactors?.additives?.tatlandirici || false,
          }
        },
        buyLink: product.buyLink || "",
        icerik: product.icerik || [],
        icerik_bilgisi: product.icerik_bilgisi || {},
        usage: product.usage || "",
        createdAt: product.createdAt || new Date()
      }))

      downloadJson(exportData, `products-${new Date().toISOString().split('T')[0]}.json`)
    } catch (error) {
      console.error("Export hatası:", error)
      alert("Export sırasında bir hata oluştu.")
    } finally {
      setExporting(false)
    }
  }

  const exportBrands = () => {
    setExporting(true)
    try {
      const exportData = brands.map(brand => ({
        id: brand.id,
        name: brand.name,
        logoUrl: brand.logoUrl,
        countryCode: brand.countryCode || "TR",
        stars: brand.stars,
        likes: brand.likes,
        dislikes: brand.dislikes,
        bio: brand.bio || []
      }))

      downloadJson(exportData, `brands-${new Date().toISOString().split('T')[0]}.json`)
    } catch (error) {
      console.error("Export hatası:", error)
      alert("Export sırasında bir hata oluştu.")
    } finally {
      setExporting(false)
    }
  }

  const exportFacilities = () => {
    setExporting(true)
    try {
      const exportData = facilities.map(facility => ({
        id: facility.id,
        name: facility.name,
        logoUrl: facility.logoUrl,
        bio: facility.bio || [],
        certificates: facility.certificates || {},
        brandsProduced: facility.brandsProduced || [],
        score: facility.score,
        createdAt: facility.createdAt || new Date(),
      }))
      downloadJson(exportData, `facilities-${new Date().toISOString().split('T')[0]}.json`)
    } catch (error) {
      console.error("Export hatası:", error)
      alert("Export sırasında bir hata oluştu.")
    } finally {
      setExporting(false)
    }
  }

  const exportAll = () => {
    setExporting(true)
    try {
      const exportData = {
        products: products.map(product => ({
          id: product.id,
          name: product.name,
          brand: product.brand,
          category: product.category,
          subCategory: product.subCategory,
          price: product.price,
          image: product.image,
          rating: product.rating,
          likes: product.likes,
          dislikes: product.dislikes,
          evaluation: {
            temiz_icerik: product.evaluation?.temiz_icerik || 0,
            formul_kalitesi: product.evaluation?.formul_kalitesi || 0,
            uretim_tesisi: product.evaluation?.uretim_tesisi || 0,
            kullanici_yorumlari: product.evaluation?.kullanici_yorumlari || 0,
            uretim_sertifika: product.evaluation?.uretim_sertifika || 0,
            ai_guven_skoru: product.evaluation?.ai_guven_skoru || 0,
            dradvice: product.evaluation?.dradvice || false,
            certFactors: {
              gmp: product.evaluation?.certFactors?.gmp || false,
              haccp: product.evaluation?.certFactors?.haccp || false,
              iso9001: product.evaluation?.certFactors?.iso9001 || false,
              iso22000: product.evaluation?.certFactors?.iso22000 || false,
            }
          },
          cleanFactors: {
            form: {
              tablet: product.cleanFactors?.form?.tablet || false,
              kapsul: product.cleanFactors?.form?.kapsul || false,
              toz: product.cleanFactors?.form?.toz || false,
              sase: product.cleanFactors?.form?.sase || false,
              gummy: product.cleanFactors?.form?.gummy || false,
              sivi: product.cleanFactors?.form?.sivi || false,
            },
            additives: {
              renklendirici: product.cleanFactors?.additives?.renklendirici || false,
              aroma: product.cleanFactors?.additives?.aroma || false,
              tatlandirici: product.cleanFactors?.additives?.tatlandirici || false,
            }
          },
          buyLink: product.buyLink || "",
          icerik: product.icerik || [],
          icerik_bilgisi: product.icerik_bilgisi || {},
          usage: product.usage || "",
          createdAt: product.createdAt || new Date(),
        })),
        brands: brands.map(brand => ({
          id: brand.id,
          name: brand.name,
          logoUrl: brand.logoUrl,
          countryCode: brand.countryCode || "TR",
          stars: brand.stars,
          likes: brand.likes,
          dislikes: brand.dislikes,
          bio: brand.bio || []
        })),
        facilities: facilities.map(facility => ({
          id: facility.id,
          name: facility.name,
          logoUrl: facility.logoUrl,
          bio: facility.bio || [],
          certificates: facility.certificates || {},
          brandsProduced: facility.brandsProduced || [],
          score: facility.score,
          createdAt: facility.createdAt || new Date(),
        })),
        exportDate: new Date().toISOString(),
        totalProducts: products.length,
        totalBrands: brands.length,
        totalFacilities: facilities.length,
        version: "1.1",
        description: "Vitamin & Takviye Ürünleri, Markalar ve Üretim Tesisleri Veri Seti"
      }
      downloadJson(exportData, `vitamin-takviye-data-${new Date().toISOString().split('T')[0]}.json`)
    } catch (error) {
      console.error("Export hatası:", error)
      alert("Export sırasında bir hata oluştu.")
    } finally {
      setExporting(false)
    }
  }

  const handleFileSelect = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0]
    if (file && file.type === "application/json") {
      setSelectedFile(file)
    } else {
      alert("Lütfen geçerli bir JSON dosyası seçin.")
    }
  }

  const importJson = async () => {
    if (!selectedFile) {
      alert("Lütfen bir dosya seçin.")
      return
    }

    setImporting(true)

    try {
      const text = await selectedFile.text()
      const data = JSON.parse(text)
      
      let importedProducts = 0
      let importedBrands = 0
      let importedFacilities = 0

      // Veri tipini kontrol et
      if (Array.isArray(data)) {
        // Array formatı - birden fazla ürün veya marka
        if (data.length > 0) {
          const firstItem = data[0]
          
          // Ürün kontrolü - category alanı varsa ürün
          if (firstItem.category) {
            // Ürün verisi
            for (const product of data) {
              try {
                // Undefined değerleri filtrele
                const cleanProduct = Object.fromEntries(
                  Object.entries(product).filter(([_, value]) => value !== undefined)
                ) as Omit<Product, "id"> & { id?: string }
                await saveProduct(cleanProduct)
                importedProducts++
              } catch (error) {
                console.error(`Ürün "${product.name}" eklenirken hata:`, error)
              }
            }
          } 
          // Marka kontrolü - bio alanı varsa ve array ise marka
          else if (firstItem.bio && Array.isArray(firstItem.bio)) {
            // Marka verisi
            for (const brand of data) {
              try {
                // Undefined değerleri filtrele
                const cleanBrand = Object.fromEntries(
                  Object.entries(brand).filter(([_, value]) => value !== undefined)
                ) as Omit<Brand, "id"> & { id?: string }
                await saveBrand(cleanBrand)
                importedBrands++
              } catch (error) {
                console.error(`Marka "${brand.name}" eklenirken hata:`, error)
              }
            }
          } else {
            throw new Error("Geçersiz veri formatı - ürün veya marka verisi bulunamadı")
          }
        }
      } else {
        throw new Error("JSON dosyası array formatında olmalıdır")
      }

      // Tesis kontrolü - certificates ve brandsProduced varsa facility
      if (data.length > 0 && (data[0].certificates || data[0].brandsProduced)) {
        for (const facility of data) {
          try {
            const cleanFacility = Object.fromEntries(
              Object.entries(facility).filter(([_, value]) => value !== undefined)
            ) as Omit<Facility, "id"> & { id?: string }
            await saveFacility(cleanFacility)
            importedFacilities++
          } catch (error) {
            console.error(`Tesis "${facility.name}" eklenirken hata:`, error)
          }
        }
      }

      alert(`${importedProducts} ürün, ${importedBrands} marka ve ${importedFacilities} tesis başarıyla içe aktarıldı.`)
      onRefresh() // Veriyi yenile
      setSelectedFile(null)
      
      // File input'u temizle
      const fileInput = document.getElementById('file-input') as HTMLInputElement
      if (fileInput) fileInput.value = ''

    } catch (error) {
      console.error("Import hatası:", error)
      alert(`JSON dosyası okunurken hata oluştu: ${error}`)
    } finally {
      setImporting(false)
    }
  }

  return (
    <div className="space-y-6">
      {/* Export Card */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Database className="h-5 w-5" />
            JSON Export
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            {/* İstatistikler */}
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
              <div className="text-center p-4 bg-blue-50 rounded-lg">
                <div className="text-2xl font-bold text-blue-600">{products.length}</div>
                <div className="text-sm text-gray-600">Toplam Ürün</div>
              </div>
              <div className="text-center p-4 bg-green-50 rounded-lg">
                <div className="text-2xl font-bold text-green-600">{brands.length}</div>
                <div className="text-sm text-gray-600">Toplam Marka</div>
              </div>
              <div className="text-center p-4 bg-yellow-50 rounded-lg">
                <div className="text-2xl font-bold text-yellow-600">
                  {products.filter(p => p.evaluation?.dradvice).length}
                </div>
                <div className="text-sm text-gray-600">Doktor Tavsiyesi</div>
              </div>
              <div className="text-center p-4 bg-purple-50 rounded-lg">
                <div className="text-2xl font-bold text-purple-600">
                  {products.filter(p => p.evaluation?.certFactors).length}
                </div>
                <div className="text-sm text-gray-600">Sertifikalı</div>
              </div>
            </div>

            {/* Export Buttons */}
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              <Button 
                onClick={exportProducts} 
                disabled={exporting}
                className="flex items-center gap-2"
                variant="outline"
              >
                <FileText className="h-4 w-4" />
                {exporting ? "Exporting..." : "Ürünleri İndir"}
              </Button>

              <Button 
                onClick={exportBrands} 
                disabled={exporting}
                className="flex items-center gap-2"
                variant="outline"
              >
                <FileText className="h-4 w-4" />
                {exporting ? "Exporting..." : "Markaları İndir"}
              </Button>

              <Button 
                onClick={exportFacilities} 
                disabled={exporting}
                className="flex items-center gap-2"
                variant="outline"
              >
                <FileText className="h-4 w-4" />
                {exporting ? "Exporting..." : "Tesisleri İndir"}
              </Button>

              <Button 
                onClick={exportAll} 
                disabled={exporting}
                className="flex items-center gap-2"
              >
                <Download className="h-4 w-4" />
                {exporting ? "Exporting..." : "Tümünü İndir"}
              </Button>
            </div>

            {/* Format Bilgisi */}
            <div className="mt-6 p-4 bg-gray-50 rounded-lg">
              <h4 className="font-medium mb-2">Export Format Bilgisi:</h4>
              <ul className="text-sm text-gray-600 space-y-1">
                <li>• <strong>Ürünler:</strong> Tüm ürün verileri (skorlar, sertifikalar, temizlik faktörleri dahil)</li>
                <li>• <strong>Markalar:</strong> Tüm marka verileri (logo, puanlar, bio dahil)</li>
                <li>• <strong>Tümünü İndir:</strong> Hem ürünler hem markalar hem tesisler tek dosyada + metadata</li>
                <li>• <strong>Detaylı Alanlar:</strong> evaluation, cleanFactors, icerik, usage, buyLink</li>
                <li>• Dosyalar JSON formatında ve tarih damgalı olarak indirilir</li>
              </ul>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Import Card */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Upload className="h-5 w-5" />
            JSON Import
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            {/* File Selection */}
            <div className="space-y-2">
              <label htmlFor="file-input" className="text-sm font-medium">
                JSON Dosyası Seçin
              </label>
              <input
                id="file-input"
                type="file"
                accept=".json"
                onChange={handleFileSelect}
                className="block w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-sm file:font-semibold file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100"
              />
              {selectedFile && (
                <p className="text-sm text-green-600">
                  ✓ {selectedFile.name} seçildi ({(selectedFile.size / 1024).toFixed(1)} KB)
                </p>
              )}
            </div>

            {/* Import Button */}
            <Button 
              onClick={importJson} 
              disabled={!selectedFile || importing}
              className="flex items-center gap-2"
            >
              <Upload className="h-4 w-4" />
              {importing ? "İçe Aktarılıyor..." : "JSON Dosyasını İçe Aktar"}
            </Button>

            {/* Import Format Bilgisi */}
            <div className="mt-6 p-4 bg-blue-50 rounded-lg">
              <h4 className="font-medium mb-2">Import Format Bilgisi:</h4>
              <ul className="text-sm text-gray-700 space-y-1">
                <li>• <strong>Array Formatı:</strong> JSON dosyası array formatında olmalıdır</li>
                <li>• <strong>Tek Tip Veri:</strong> Sadece ürünler VEYA sadece markalar içerebilir</li>
                <li>• <strong>Otomatik Tanıma:</strong> İlk elemanın özelliklerine göre tip belirlenir</li>
                <li>• <strong>Marka Alanları:</strong> name, logoUrl, countryCode, stars, likes, dislikes, bio</li>
                <li>• <strong>Logo URL:</strong> Geçerli bir URL olmalı (https://example.com/logo.png)</li>
                <li>• <strong>Bio Formatı:</strong> Array of objects: {'[{'}"content": "...", "reference": "..."{'}]'}</li>
                <li>• <strong>Hata Yönetimi:</strong> Başarısız olanlar atlanır, başarılı olanlar eklenir</li>
                <li>• <strong>Mevcut Veriler:</strong> Aynı ID'ye sahip veriler güncellenir</li>
              </ul>
              
              {/* Örnek Format */}
              <details className="group mt-4">
                <summary className="cursor-pointer text-sm font-medium text-gray-700 hover:text-gray-900">
                  Örnek Import Formatını Göster
                </summary>
                <div className="mt-2 p-4 bg-gray-50 rounded-lg text-xs font-mono overflow-x-auto">
                  <pre>{`// Ürünler Array Formatı
[
  {
    "name": "Vitamin D3 1000 IU",
    "brand": "Ava Vitamin",
    "category": "Vitamin & Mineral",
    "price": 89.99,
    "evaluation": {
      "ai_guven_skoru": 85,
      "certFactors": { "gmp": true }
    },
    "cleanFactors": {
      "form": { "kapsul": true },
      "additives": { "aroma": false }
    },
    "icerik": ["D3 Vitamini", "Zeytinyağı"],
    "usage": "Günde 1 kapsül"
  }
]

// Markalar Array Formatı
[
  {
    "name": "Kiperin",
    "logoUrl": "https://example.com/kiperin-logo.png",
    "countryCode": "TR",
    "stars": 4.5,
    "likes": 150,
    "dislikes": 10,
    "bio": [
      {
        "content": "Yerli üretim yapan güvenilir marka",
        "reference": "https://www.kiperin.com.tr"
      },
      {
        "content": "GMP sertifikasına sahip üretim tesisi"
      }
    ]
  },
  {
    "name": "Ava Vitamin",
    "logoUrl": "https://example.com/ava-vitamin-logo.jpg",
    "countryCode": "TR",
    "stars": 4.2,
    "likes": 200,
    "dislikes": 50,
    "bio": [
      {
        "content": "Türkiye'nin önde gelen vitamin markası",
        "reference": "https://www.avavitamin.com.tr"
      },
      {
        "content": "Geniş ürün yelpazesi ve uygun fiyatlar"
      }
    ]
  }
]`}</pre>
                </div>
              </details>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  )
} 