"use client"

import { Responsive, WidthProvider } from "react-grid-layout";
import "react-grid-layout/css/styles.css";

import { useState, useEffect } from "react";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Product, HomePageSection, CustomWidgetTemplate } from "@/types";
import { Plus, Trash2, Settings } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Label } from "@/components/ui/label";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogFooter } from "@/components/ui/dialog";
import ProductSelect from "@/components/admin/ProductSelect";
import { loadCustomWidgetTemplates } from "@/lib/firebase";

const ResponsiveGridLayout = WidthProvider(Responsive);

interface HomePageLayoutTabProps {
  products: Product[];
  onSaveLayout: (layout: HomePageSection[]) => Promise<void>;
  onGetLayout: () => Promise<HomePageSection[]>;
}

const HomePageLayoutTab = ({
  products,
  onSaveLayout,
  onGetLayout,
}: HomePageLayoutTabProps) => {
  const [layout, setLayout] = useState<HomePageSection[]>([]);
  const [customWidgetTemplates, setCustomWidgetTemplates] = useState<CustomWidgetTemplate[]>([]);
  const [loading, setLoading] = useState(true);
  const [isEditMode, setIsEditMode] = useState(false); // New state for edit mode

  useEffect(() => {
    const fetchCustomWidgetTemplates = async () => {
      try {
        const data = await loadCustomWidgetTemplates();
        setCustomWidgetTemplates(data);
      } catch (error) {
        console.error("Özel widget şablonları yüklenirken hata:", error);
      }
    };
    fetchCustomWidgetTemplates();
  }, []);
  const [isEditModalOpen, setIsEditModalOpen] = useState(false);
  const [currentEditingSection, setCurrentEditingSection] = useState<HomePageSection | null>(null);

  const onLayoutChange = (newLayout: any[]) => {
    const updatedLayout = layout.map(section => {
      const layoutItem = newLayout.find(item => item.i === section.id);
      if (layoutItem) {
        return {
          ...section,
          x: layoutItem.x,
          y: layoutItem.y,
          w: layoutItem.w,
          h: layoutItem.h,
        };
      }
      return section;
    });
    saveLayout(updatedLayout);
  };

  useEffect(() => {
    const fetchLayout = async () => {
      setLoading(true);
      try {
        const layoutData = await onGetLayout();
        // Ensure all layout items have x, y, w, h properties
        const initializedLayout = layoutData.map(section => ({
          ...section,
          x: section.x !== undefined ? section.x : 0,
          y: section.y !== undefined ? section.y : 0,
          w: section.w !== undefined ? section.w : 2,
          h: section.h !== undefined ? section.h : 1,
        }));
        setLayout(initializedLayout);
        console.log("Initialized Layout:", initializedLayout); // Debugging log
      } catch (error) {
        console.error("Layout yüklenirken hata:", error);
      } finally {
        setLoading(false);
      }
    };
    fetchLayout();
  }, [onGetLayout]);

  const saveLayout = async (newLayout: HomePageSection[]) => {
    try {
      await onSaveLayout(newLayout);
      setLayout(newLayout);
    } catch (error) {
      console.error("Layout kaydedilirken hata:", error);
    }
  };

  const addSection = (type: HomePageSection["type"]) => {
    const newSection: HomePageSection = {
      id: new Date().toISOString(),
      type: type,
      title: `Yeni Bölüm (${type})`,
      order: layout.length,
      x: (layout.length * 2) % 6,
      y: Math.floor(layout.length / 3),
      w: 1, // Default width, will be overridden by custom widget template
      h: 1, // Default height, will be overridden by custom widget template
    };
    const updatedLayout = [...layout, newSection];
    saveLayout(updatedLayout);
  };

  const removeSection = (id: string) => {
    const updatedLayout = layout.filter(section => section.id !== id);
    saveLayout(updatedLayout);
  };

  const updateSection = (id: string, updates: Partial<HomePageSection>) => {
    const updatedLayout = layout.map(section => 
      section.id === id ? { ...section, ...updates } : section
    );
    saveLayout(updatedLayout);
  };

  const handleEditSection = (section: HomePageSection) => {
    setCurrentEditingSection(section);
    setIsEditModalOpen(true);
  };

  const handleSaveEdit = () => {
    if (currentEditingSection) {
      updateSection(currentEditingSection.id, currentEditingSection);
      setIsEditModalOpen(false);
      setCurrentEditingSection(null);
    }
  };

  if (loading) {
    return <div>Yükleniyor...</div>;
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle>Anasayfa Düzeni Yönetimi</CardTitle>
        <div className="flex gap-2 mt-4">
          <Button onClick={() => addSection("custom_widget")}>
            <Plus className="mr-2 h-4 w-4" /> Özel Widget Ekle
          </Button>
          <Button onClick={() => addSection("all_products")}>
            <Plus className="mr-2 h-4 w-4" /> Tüm Ürünler Listesi Ekle
          </Button>
          <Button onClick={() => setIsEditMode(!isEditMode)} variant={isEditMode ? "destructive" : "default"}>
            {isEditMode ? "Düzenleme Modundan Çık" : "Düzenleme Moduna Gir"}
          </Button>
        </div>
      </CardHeader>
      <CardContent className="space-y-4">
        <ResponsiveGridLayout
          className="layout"
          layouts={{ lg: layout.map(section => ({ i: section.id, x: section.x, y: section.y, w: section.w, h: section.h })) }}
          breakpoints={{ lg: 1200, md: 996, sm: 768, xs: 480, xxs: 0 }}
          cols={{ lg: 6, md: 6, sm: 6, xs: 4, xxs: 2 }}
          rowHeight={200}
          margin={[0, 0]} // Add this line to remove gaps
          isDraggable={isEditMode} // Enable dragging only in edit mode
          isResizable={isEditMode} // Enable resizing only in edit mode
          onLayoutChange={(currentLayout) => onLayoutChange(currentLayout)}
        >
          {layout.map((section) => (
            <div key={section.id} className="border rounded-lg p-4 bg-white shadow-sm flex flex-col justify-between">
              <div className="flex-1 space-y-2">
                <h3 className="font-semibold text-lg">{section.title || "Başlıksız Bölüm"}</h3>
                <p className="text-sm text-gray-500">Tip: {section.type}</p>
                <p className="text-sm text-gray-500">Boyut: {section.w}x{section.h} Konum: ({section.x},{section.y})</p>
              </div>
              <div className="mt-auto flex justify-end gap-2">
                <Button size="icon" variant="outline" onClick={() => handleEditSection(section)}>
                  <Settings className="h-4 w-4" />
                </Button>
                <Button size="icon" variant="destructive" onClick={() => removeSection(section.id)}>
                  <Trash2 className="h-4 w-4" />
                </Button>
              </div>
            </div>
          ))}
        </ResponsiveGridLayout>
        {layout.length === 0 && (
          <p className="text-center text-gray-500 py-8">Henüz anasayfa bölümü eklenmemiş.</p>
        )}
      </CardContent>

      <Dialog open={isEditModalOpen} onOpenChange={setIsEditModalOpen}>
        <DialogContent className="sm:max-w-[600px]">
          <DialogHeader>
            <DialogTitle>Bölümü Düzenle</DialogTitle>
          </DialogHeader>
          {currentEditingSection && (
            <div className="grid gap-4 py-4">
              <div className="grid grid-cols-4 items-center gap-4">
                <Label htmlFor="title" className="text-right">
                  Başlık
                </Label>
                <Input
                  id="title"
                  value={currentEditingSection.title || ""}
                  onChange={(e) => setCurrentEditingSection({ ...currentEditingSection, title: e.target.value })}
                  className="col-span-3"
                />
              </div>

              <div className="grid grid-cols-4 items-center gap-4">
                <Label htmlFor="type" className="text-right">
                  Bölüm Tipi
                </Label>
                <Input
                  id="type"
                  value={currentEditingSection.type}
                  className="col-span-3"
                  disabled
                />
              </div>

              {currentEditingSection.type === "product_carousel" && (
                <>
                  <div className="grid grid-cols-4 items-center gap-4">
                    <Label htmlFor="bgColor" className="text-right">
                      Arka Plan Rengi
                    </Label>
                    <Input
                      id="bgColor"
                      value={currentEditingSection.bgColor || ""}
                      onChange={(e) => setCurrentEditingSection({ ...currentEditingSection, bgColor: e.target.value })}
                      className="col-span-3"
                      placeholder="Tailwind sınıfı, örn: bg-blue-100"
                    />
                  </div>
                  <div className="grid grid-cols-4 items-center gap-4">
                    <Label htmlFor="filter" className="text-right">
                      Karosel Filtresi
                    </Label>
                    <Select onValueChange={(value) => setCurrentEditingSection({ ...currentEditingSection, filter: value as any })} value={currentEditingSection.filter}>
                      <SelectTrigger className="col-span-3"><SelectValue placeholder="Karosel Filtresini Seç" /></SelectTrigger>
                      <SelectContent>
                        <SelectItem value="most_liked">En Beğenilenler</SelectItem>
                        <SelectItem value="newest">En Yeniler</SelectItem>
                        <SelectItem value="dradvice_products">Doktor Tavsiyeleri</SelectItem>
                        <SelectItem value="custom_products">Özel Ürün Seçimi</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                  {currentEditingSection.filter === "custom_products" && (
                    <div className="grid grid-cols-4 items-start gap-4">
                      <Label htmlFor="productIds" className="text-right pt-2">
                        Ürünler
                      </Label>
                      <div className="col-span-3">
                        <ProductSelect
                          allProducts={products}
                          selectedProductIds={currentEditingSection.productIds || []}
                          onProductIdsChange={(ids) => setCurrentEditingSection({ ...currentEditingSection, productIds: ids })}
                        />
                      </div>
                    </div>
                  )}
                </>
              )}

              {currentEditingSection.type === "custom_widget" && (
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="sectionId" className="text-right">
                    Widget Şablonu
                  </Label>
                  <Select onValueChange={(value) => {
                    const selectedTemplate = customWidgetTemplates.find(t => t.id === value);
                    if (selectedTemplate) {
                      setCurrentEditingSection({
                        ...currentEditingSection,
                        sectionId: value,
                        w: selectedTemplate.w || currentEditingSection.w, // Update width from template
                        h: selectedTemplate.h || currentEditingSection.h, // Update height from template
                        title: selectedTemplate.title || currentEditingSection.title, // Update title from template
                        description: selectedTemplate.description || currentEditingSection.description, // Update description from template
                      });
                    } else {
                      setCurrentEditingSection({ ...currentEditingSection, sectionId: value });
                    }
                  }} value={currentEditingSection.sectionId}>
                    <SelectTrigger className="col-span-3"><SelectValue placeholder="Widget Şablonu Seç" /></SelectTrigger>
                    <SelectContent>
                      {customWidgetTemplates.map(template => <SelectItem key={template.id} value={template.id}>{template.name}</SelectItem>)}
                    </SelectContent>
                  </Select>
                </div>
              )}

              {/* Grid Ayarları */}
              <div className="grid grid-cols-4 items-center gap-4">
                <Label htmlFor="w" className="text-right">
                  Sütun Genişliği (1-4)
                </Label>
                <Input
                  id="w"
                  type="number"
                  min={1}
                  max={4}
                  value={currentEditingSection.w}
                  onChange={(e) => setCurrentEditingSection({ ...currentEditingSection, w: parseInt(e.target.value) })}
                  className="col-span-3"
                />
              </div>
              <div className="grid grid-cols-4 items-center gap-4">
                <Label htmlFor="h" className="text-right">
                  Satır Yüksekliği
                </Label>
                <Input
                  id="h"
                  type="number"
                  min={1}
                  value={currentEditingSection.h}
                  onChange={(e) => setCurrentEditingSection({ ...currentEditingSection, h: parseInt(e.target.value) })}
                  className="col-span-3"
                />
              </div>
            </div>
          )}
          <DialogFooter>
            <Button variant="outline" onClick={() => setIsEditModalOpen(false)}>
              İptal
            </Button>
            <Button onClick={handleSaveEdit}>
              Kaydet
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </Card>
  );
};

export default HomePageLayoutTab;