"use client"

import { useState, useCallback } from "react"
import { Search, Edit, Trash2, Plus, Award, Zap } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import type { Facility } from "@/types"
import Image from "next/image"
import { deleteFacility, saveFacility } from "@/lib/firebase"
import { Badge } from "@/components/ui/badge"
import { Checkbox } from "@/components/ui/checkbox"

interface FacilitiesTabProps {
  facilities: Facility[]
  products: any[]
  brands: any[]
  loading: boolean
  onEdit: (facility: Facility) => void
  onDelete: (id: string) => void
  onAdd: () => void
  onRefresh: () => void
}

const QUICK_EDIT_CERTIFICATES = [
  { value: "gmp", label: "GMP" },
  { value: "haccp", label: "HACCP" },
  { value: "iso9001", label: "ISO 9001" },
  { value: "iso22000", label: "ISO 22000" },
];

export default function FacilitiesTab({ facilities, products, brands, loading, onEdit, onDelete, onAdd, onRefresh }: FacilitiesTabProps) {
  const [searchQuery, setSearchQuery] = useState("")
  const [deleting, setDeleting] = useState<string | null>(null)
  const [selectedIds, setSelectedIds] = useState<string[]>([])
  const [syncing, setSyncing] = useState(false)
  const [editingCertificates, setEditingCertificates] = useState(false)

  const handleSync = async () => {
    setSyncing(true)
    try {
      await Promise.all(facilities.map(async facility => {
        const normalizeKayitNo = (obj: any) => {
          if (!obj) return '';
          let kayitNo = '';
          if (typeof obj === 'object') {
            kayitNo = obj.kayit_no || obj.onayNo || obj["Üretici İşletme Kayıt/Onay Numarası"] || obj.uretimTesisiKayitNo || '';
          } else if (typeof obj === 'string') {
            kayitNo = obj;
          }
          // Extract the TR-XX-K- part for matching
          // Extract the part before the last hyphen, or the whole string if no hyphen
          const lastHyphenIndex = kayitNo.lastIndexOf('-');
          return (lastHyphenIndex !== -1 ? kayitNo.substring(0, lastHyphenIndex) : kayitNo).toLowerCase();
        };

        const tesisKod = normalizeKayitNo(facility);
        console.log("Tesis Kodu:", tesisKod, "Facility:", facility);

        const matchedProducts = products.filter(p => {
          const urunKod = normalizeKayitNo(p);
          console.log("Ürün Kodu:", urunKod, "Product:", p);
          return urunKod && tesisKod && urunKod === tesisKod;
        });
        console.log("Eşleşen Ürünler:", matchedProducts);

        const brandsProduced = Array.from(new Set(
          matchedProducts.map(p => {
            if (typeof p.brand === 'string') {
              return p.brand;
            }
            if (typeof p.brand === 'object' && p.brand !== null && (p.brand as any).name) {
              return (p.brand as any).name;
            }
            return null;
          }).filter(Boolean)
        )) as string[];

        // @ts-ignore
        await saveFacility({ ...facility, brandsProduced });
      }));

      onRefresh();
      alert("Eşleştirme tamamlandı!")
    } catch (error) {
      console.error("Eşleştirme hatası:", error)
      alert("Eşleştirme sırasında bir hata oluştu.")
    } finally {
      setSyncing(false)
    }
  }

  const filteredFacilities = facilities.filter((facility) => facility.name.toLowerCase().includes(searchQuery.toLowerCase()))

  const handleDelete = async (id: string) => {
    if (!confirm("Bu tesisi silmek istediğinizden emin misiniz?")) return
    try {
      setDeleting(id)
      await deleteFacility(id)
      onDelete(id)
      onRefresh()
    } catch (error) {
      console.error("Tesis silinirken hata:", error)
      alert("Tesis silinirken bir hata oluştu.")
    } finally {
      setDeleting(null)
    }
  }

  const handleSelect = (id: string) => {
    setSelectedIds((prev) => prev.includes(id) ? prev.filter((sid) => sid !== id) : [...prev, id])
  }

  const handleSelectAll = () => {
    if (selectedIds.length === filteredFacilities.length) {
      setSelectedIds([])
    } else {
      setSelectedIds(filteredFacilities.map((f) => f.id))
    }
  }

  const handleQuickEditSave = async () => {
    if (!selectedIds.length || !quickEditCertificate) return;

    try {
      await Promise.all(selectedIds.map(async (id) => {
        const facility = facilities.find(f => f.id === id);
        if (facility) {
          const updatedCertificates = {
            ...facility.certificates,
            [quickEditCertificate]: !(facility.certificates && facility.certificates[quickEditCertificate])
          };
          // @ts-ignore
          await saveFacility({ ...facility, certificates: updatedCertificates });
        }
      }));
      setSelectedIds([]);
      onRefresh();
      alert("Sertifikalar başarıyla güncellendi!");
    } catch (error) {
      console.error("Sertifika hızlı düzenleme hatası:", error);
      alert("Sertifika hızlı düzenleme sırasında bir hata oluştu.");
    }
  };

  const propagateCertificates = async (updatedFacility: Facility) => {
    // 1. Markaları güncelle
    const brandsToUpdate = brands.filter(brand => updatedFacility.brandsProduced?.includes(brand.name));

    await Promise.all(brandsToUpdate.map(async brand => {
      const mergedCertificates = { ...brand.certificates, ...updatedFacility.certificates };
      // @ts-ignore
      await saveBrand({ ...brand, certificates: mergedCertificates });

      // 2. Ürünleri güncelle (sadece bu markaya ait olanları)
      const productsToUpdate = products.filter(product => {
        if (typeof product.brand === 'string') {
          return product.brand === brand.name;
        } else if (typeof product.brand === 'object' && product.brand !== null) {
          return product.brand.id === brand.id || product.brand.name === brand.name;
        }
        return false;
      });

      await Promise.all(productsToUpdate.map(async product => {
        const updatedEvaluation = {
          ...product.evaluation,
          certFactors: { ...product.evaluation?.certFactors, ...mergedCertificates }
        };
        // @ts-ignore
        await saveProduct({ ...product, evaluation: updatedEvaluation });
      }));
    }));
  };

  const handleBulkDelete = async () => {
    if (!selectedIds.length) return
    if (!confirm(`${selectedIds.length} tesisi silmek istediğinize emin misiniz?`)) return
    try {
      await Promise.all(selectedIds.map(async (id) => {
        await deleteFacility(id)
        onDelete(id)
      }))
      setSelectedIds([])
      onRefresh()
    } catch (error) {
      alert("Toplu silme sırasında hata oluştu.")
    }
  }

  if (loading) {
    return (
      <Card>
        <CardContent className="flex items-center justify-center py-20">
          <div className="text-center">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto mb-4"></div>
            <p className="text-gray-600">Tesisler yükleniyor...</p>
          </div>
        </CardContent>
      </Card>
    )
  }

  return (
    <div className="space-y-6">
      {/* Controls */}
      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <CardTitle>Üretim Tesisleri ({filteredFacilities.length})</CardTitle>
            <div className="flex gap-2">
              <Button variant="destructive" onClick={handleBulkDelete} disabled={!selectedIds.length}>
                Seçilenleri Sil
              </Button>
              <Button onClick={() => setEditingCertificates(!editingCertificates)} variant="outline">
                <Award className="h-4 w-4 mr-2" />
                {editingCertificates ? "Düzenlemeyi Bitir" : "Sertifikaları Düzenle"}
              </Button>
              <Button onClick={onAdd}>
                <Plus className="h-4 w-4 mr-2" />
                Yeni Tesis
              </Button>
              <Button onClick={handleSync} variant="outline">
                <Zap className="h-4 w-4 mr-2" />
                Ürün/Marka Eşleştir
              </Button>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          <div className="relative flex items-center gap-2 mb-2">
            <Checkbox
              checked={selectedIds.length === filteredFacilities.length && filteredFacilities.length > 0
                ? true
                : selectedIds.length > 0
                ? 'indeterminate'
                : false}
              onCheckedChange={handleSelectAll}
            />
            <span className="text-xs text-gray-500 select-none">Tümünü Seç</span>
            <Search className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
            <Input
              placeholder="Tesis adı ara..."
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              className="pl-10"
            />
          </div>
        </CardContent>
      </Card>

      {/* Facilities Grid */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        {filteredFacilities.map((facility) => (
          <Card key={facility.id} className="hover:shadow-lg transition-shadow">
            <CardContent className="p-6">
              <div className="flex items-start justify-between mb-4">
                <div className="flex items-center gap-3">
                  <Checkbox checked={selectedIds.includes(facility.id)} onCheckedChange={() => handleSelect(facility.id)} />
                  <div className="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center overflow-hidden flex-shrink-0">
                    <Image
                      src={facility.logoUrl || "/placeholder-logo.png"}
                      alt={facility.name}
                      width={64}
                      height={64}
                      className="object-contain"
                      onError={(e) => {
                        const target = e.target as HTMLImageElement
                        target.src = "/placeholder-logo.png"
                      }}
                    />
                  </div>
                  <div className="flex-1">
                    <h3 className="font-bold text-xl text-gray-900">{facility.name}</h3>
                    <div className="flex items-center gap-2 mt-1">
                      <Award className="h-5 w-5 text-yellow-500" />
                      <span className="text-sm text-gray-700">Puan: <span className="font-semibold">{facility.score}/100</span></span>
                    </div>
                  </div>
                </div>
                {editingCertificates ? (
                  <div className="flex flex-col gap-2">
                    {QUICK_EDIT_CERTIFICATES.map((cert) => (
                      <div key={cert.value} className="flex items-center space-x-2">
                        <Checkbox
                          id={`${facility.id}-${cert.value}`}
                          checked={facility.certificates && facility.certificates[cert.value]}
                          onCheckedChange={async (checked) => {
                            const updatedCertificates = {
                              ...facility.certificates,
                              [cert.value]: checked as boolean
                            };
                            // @ts-ignore
                            const savedFacilityId = await saveFacility({ ...facility, certificates: updatedCertificates });
                            const updatedFacility = { ...facility, id: savedFacilityId, certificates: updatedCertificates };
                            await propagateCertificates(updatedFacility);
                            onRefresh();
                          }}
                        />
                        <label
                          htmlFor={`${facility.id}-${cert.value}`}
                          className={`text-sm font-medium ${facility.certificates && facility.certificates[cert.value] ? "text-blue-600" : "text-gray-700"}`}
                        >
                          {cert.label}
                        </label>
                      </div>
                    ))}
                  </div>
                ) : (
                  <div className="flex items-center gap-2">
                    <Button size="sm" variant="outline" onClick={() => onEdit(facility)}>
                      <Edit className="h-4 w-4" />
                    </Button>
                    <Button
                      size="sm"
                      variant="outline"
                      className="text-red-600 hover:text-red-700"
                      onClick={() => handleDelete(facility.id)}
                      disabled={deleting === facility.id}
                    >
                      <Trash2 className="h-4 w-4" />
                    </Button>
                  </div>
                )}
              </div>

              {/* Certificates Display */}
              {!editingCertificates && facility.certificates && (
                <div className="mt-4 mb-4">
                  <h4 className="font-semibold text-sm mb-2">Sertifikalar:</h4>
                  <div className="flex flex-wrap gap-2">
                    {QUICK_EDIT_CERTIFICATES.map((cert) => (
                      <div key={cert.value} className="flex items-center gap-1">
                        {facility.certificates[cert.value] ? (
                          <Check className="h-4 w-4 text-green-500" />
                        ) : (
                          <X className="h-4 w-4 text-gray-400" />
                        )}
                        <span className={`text-xs font-medium ${facility.certificates[cert.value] ? "text-green-700" : "text-gray-500"}`}>
                          {cert.label}
                        </span>
                      </div>
                    ))}
                  </div>
                </div>
              )}

              {/* Brands Produced */}
              {facility.brandsProduced && facility.brandsProduced.length > 0 && (
                <div className="mt-4">
                  <h4 className="font-semibold text-sm mb-2">Üretilen Markalar:</h4>
                  <div className="flex flex-wrap gap-2">
                    {facility.brandsProduced.map((brandName) => {
                      const brand = brands.find(b => b.name === brandName);
                      return (
                        <Badge key={brandName} variant="secondary" className="text-xs flex items-center gap-1">
                          {brand?.logoUrl && (
                            <Image src={brand.logoUrl} alt={brand.name} width={16} height={16} className="rounded-full" />
                          )}
                          {brandName}
                        </Badge>
                      );
                    })}
                  </div>
                </div>
              )}

              {/* Bio */}
              {facility.bio && facility.bio.length > 0 && (
                <div className="mt-4">
                  <h4 className="font-semibold text-sm mb-2">Hakkında:</h4>
                  <ul className="text-sm text-gray-600 space-y-2">
                    {facility.bio.slice(0, 3).map((item, index) => (
                      <li key={index} className="flex items-start gap-2">
                        <span className="text-blue-600 mt-1">•</span>
                        <div className="flex-1">
                          <span>{item.content}</span>
                          {item.reference && (
                            <div className="mt-1">
                              <a
                                href={item.reference}
                                target="_blank"
                                rel="noopener noreferrer"
                                className="text-xs text-blue-600 hover:text-blue-800 underline"
                                title="Detaylar için tıklayınız"
                              >
                                📎 Kaynak
                              </a>
                            </div>
                          )}
                        </div>
                      </li>
                    ))}
                  </ul>
                </div>
              )}
            </CardContent>
          </Card>
        ))}
      </div>
    </div>
  )
} 